/**
 * Provides replacement for Gtk.accelerator_parse.
 */
export class AcceleratorParse {
    constructor() {
        /**
         * GDK keystr mask values.
         * Refer to:
         * https://gitlab.gnome.org/GNOME/gtk/-/blob/4.13.0/gdk/gdkenums.h?ref_type=tags#L115
         */
        this.GDK_SHIFT_MASK    = 1 << 0;
        this.GDK_CONTROL_MASK  = 1 << 2;
        this.GDK_ALT_MASK      = 1 << 3;
        this.GDK_SUPER_MASK    = 1 << 26;
        this.GDK_HYPER_MASK    = 1 << 27;
        this.GDK_META_MASK     = 1 << 28;
    }

    accelerator_parse(keystr) {
        let mask = this.accelerator_mask(keystr);
        const mods = this.accelerator_mods(keystr);

        // remove mods from keystr
        let key = keystr;
        mods.forEach(m => {
            key = key.replace(m, '');
        });
        key = key.trim();

        // now lookup key in map
        this._initKeycodeMap();
        let ok = true;
        const mapped = this.keycodeMap.get(key);
        if (mapped) {
            key = mapped;
        }
        else {
            ok = false;
            key = 0;
            mask = 0;
        }

        // console.log(keystr, key, mask);
        return [ok, key, mask];
    }

    /**
     * Returns array of mods for a keystr, e.g. ['<Control>', '<Shift>', '<Alt>'].
     * @param {String} keystr
     */
    accelerator_mods(keystr) {
        return keystr.match(/<.*?>/g) ?? [];
    }

    /**
     * Returns the GDK mask value for a keystr (keybind string representation).
     * Refer to:
     * https://gitlab.gnome.org/GNOME/gtk/-/blob/4.13.0/gdk/gdkenums.h?ref_type=tags#L115
     * https://gitlab.gnome.org/GNOME/gtk/-/blob/4.13.0/gtk/gtkaccelgroup.c#L571
     * @param {String} keystr
     */
    accelerator_mask(keystr) {
        // need to extact all mods from keystr
        const mods = this.accelerator_mods(keystr);
        let result = 0;
        for (let mod of mods) {
            switch (mod.toLowerCase()) {
            case '<shift>':
                result |= this.GDK_SHIFT_MASK;
                break;
            case '<control>':
            case '<ctrl>':
            case '<primary>':
                result |= this.GDK_CONTROL_MASK;
                break;
            case '<alt>':
                result |= this.GDK_ALT_MASK;
                break;
            case '<super>':
                result |= this.GDK_SUPER_MASK;
                break;
            case '<hyper>':
                result |= this.GDK_HYPER_MASK;
                break;
            case '<meta>':
                result |= this.GDK_META_MASK;
            }
        }

        return result;
    }

    /**
     * Replicates Gdk full keyset.
     * https://gitlab.gnome.org/GNOME/gtk/-/blob/4.13.0/gdk/gdkkeysyms.h?ref_type=tags
     */
    _initKeycodeMap(force = false) {
        if (!force && this.keycodeMap) {
            return;
        }
        const map = new Map();
        map.set('VoidSymbol', 0xffffff);
        map.set('BackSpace', 0xff08);
        map.set('Tab', 0xff09);
        map.set('Linefeed', 0xff0a);
        map.set('Clear', 0xff0b);
        map.set('Return', 0xff0d);
        map.set('Pause', 0xff13);
        map.set('Scroll_Lock', 0xff14);
        map.set('Sys_Req', 0xff15);
        map.set('Escape', 0xff1b);
        map.set('Delete', 0xffff);
        map.set('Multi_key', 0xff20);
        map.set('Codeinput', 0xff37);
        map.set('SingleCandidate', 0xff3c);
        map.set('MultipleCandidate', 0xff3d);
        map.set('PreviousCandidate', 0xff3e);
        map.set('Kanji', 0xff21);
        map.set('Muhenkan', 0xff22);
        map.set('Henkan_Mode', 0xff23);
        map.set('Henkan', 0xff23);
        map.set('Romaji', 0xff24);
        map.set('Hiragana', 0xff25);
        map.set('Katakana', 0xff26);
        map.set('Hiragana_Katakana', 0xff27);
        map.set('Zenkaku', 0xff28);
        map.set('Hankaku', 0xff29);
        map.set('Zenkaku_Hankaku', 0xff2a);
        map.set('Touroku', 0xff2b);
        map.set('Massyo', 0xff2c);
        map.set('Kana_Lock', 0xff2d);
        map.set('Kana_Shift', 0xff2e);
        map.set('Eisu_Shift', 0xff2f);
        map.set('Eisu_toggle', 0xff30);
        map.set('Kanji_Bangou', 0xff37);
        map.set('Zen_Koho', 0xff3d);
        map.set('Mae_Koho', 0xff3e);
        map.set('Home', 0xff50);
        map.set('Left', 0xff51);
        map.set('Up', 0xff52);
        map.set('Right', 0xff53);
        map.set('Down', 0xff54);
        map.set('Prior', 0xff55);
        map.set('Page_Up', 0xff55);
        map.set('Next', 0xff56);
        map.set('Page_Down', 0xff56);
        map.set('End', 0xff57);
        map.set('Begin', 0xff58);
        map.set('Select', 0xff60);
        map.set('Print', 0xff61);
        map.set('Execute', 0xff62);
        map.set('Insert', 0xff63);
        map.set('Undo', 0xff65);
        map.set('Redo', 0xff66);
        map.set('Menu', 0xff67);
        map.set('Find', 0xff68);
        map.set('Cancel', 0xff69);
        map.set('Help', 0xff6a);
        map.set('Break', 0xff6b);
        map.set('Mode_switch', 0xff7e);
        map.set('script_switch', 0xff7e);
        map.set('Num_Lock', 0xff7f);
        map.set('KP_Space', 0xff80);
        map.set('KP_Tab', 0xff89);
        map.set('KP_Enter', 0xff8d);
        map.set('KP_F1', 0xff91);
        map.set('KP_F2', 0xff92);
        map.set('KP_F3', 0xff93);
        map.set('KP_F4', 0xff94);
        map.set('KP_Home', 0xff95);
        map.set('KP_Left', 0xff96);
        map.set('KP_Up', 0xff97);
        map.set('KP_Right', 0xff98);
        map.set('KP_Down', 0xff99);
        map.set('KP_Prior', 0xff9a);
        map.set('KP_Page_Up', 0xff9a);
        map.set('KP_Next', 0xff9b);
        map.set('KP_Page_Down', 0xff9b);
        map.set('KP_End', 0xff9c);
        map.set('KP_Begin', 0xff9d);
        map.set('KP_Insert', 0xff9e);
        map.set('KP_Delete', 0xff9f);
        map.set('KP_Equal', 0xffbd);
        map.set('KP_Multiply', 0xffaa);
        map.set('KP_Add', 0xffab);
        map.set('KP_Separator', 0xffac);
        map.set('KP_Subtract', 0xffad);
        map.set('KP_Decimal', 0xffae);
        map.set('KP_Divide', 0xffaf);
        map.set('KP_0', 0xffb0);
        map.set('KP_1', 0xffb1);
        map.set('KP_2', 0xffb2);
        map.set('KP_3', 0xffb3);
        map.set('KP_4', 0xffb4);
        map.set('KP_5', 0xffb5);
        map.set('KP_6', 0xffb6);
        map.set('KP_7', 0xffb7);
        map.set('KP_8', 0xffb8);
        map.set('KP_9', 0xffb9);
        map.set('F1', 0xffbe);
        map.set('F2', 0xffbf);
        map.set('F3', 0xffc0);
        map.set('F4', 0xffc1);
        map.set('F5', 0xffc2);
        map.set('F6', 0xffc3);
        map.set('F7', 0xffc4);
        map.set('F8', 0xffc5);
        map.set('F9', 0xffc6);
        map.set('F10', 0xffc7);
        map.set('F11', 0xffc8);
        map.set('L1', 0xffc8);
        map.set('F12', 0xffc9);
        map.set('L2', 0xffc9);
        map.set('F13', 0xffca);
        map.set('L3', 0xffca);
        map.set('F14', 0xffcb);
        map.set('L4', 0xffcb);
        map.set('F15', 0xffcc);
        map.set('L5', 0xffcc);
        map.set('F16', 0xffcd);
        map.set('L6', 0xffcd);
        map.set('F17', 0xffce);
        map.set('L7', 0xffce);
        map.set('F18', 0xffcf);
        map.set('L8', 0xffcf);
        map.set('F19', 0xffd0);
        map.set('L9', 0xffd0);
        map.set('F20', 0xffd1);
        map.set('L10', 0xffd1);
        map.set('F21', 0xffd2);
        map.set('R1', 0xffd2);
        map.set('F22', 0xffd3);
        map.set('R2', 0xffd3);
        map.set('F23', 0xffd4);
        map.set('R3', 0xffd4);
        map.set('F24', 0xffd5);
        map.set('R4', 0xffd5);
        map.set('F25', 0xffd6);
        map.set('R5', 0xffd6);
        map.set('F26', 0xffd7);
        map.set('R6', 0xffd7);
        map.set('F27', 0xffd8);
        map.set('R7', 0xffd8);
        map.set('F28', 0xffd9);
        map.set('R8', 0xffd9);
        map.set('F29', 0xffda);
        map.set('R9', 0xffda);
        map.set('F30', 0xffdb);
        map.set('R10', 0xffdb);
        map.set('F31', 0xffdc);
        map.set('R11', 0xffdc);
        map.set('F32', 0xffdd);
        map.set('R12', 0xffdd);
        map.set('F33', 0xffde);
        map.set('R13', 0xffde);
        map.set('F34', 0xffdf);
        map.set('R14', 0xffdf);
        map.set('F35', 0xffe0);
        map.set('R15', 0xffe0);
        map.set('Shift_L', 0xffe1);
        map.set('Shift_R', 0xffe2);
        map.set('Control_L', 0xffe3);
        map.set('Control_R', 0xffe4);
        map.set('Caps_Lock', 0xffe5);
        map.set('Shift_Lock', 0xffe6);
        map.set('Meta_L', 0xffe7);
        map.set('Meta_R', 0xffe8);
        map.set('Alt_L', 0xffe9);
        map.set('Alt_R', 0xffea);
        map.set('Super_L', 0xffeb);
        map.set('Super_R', 0xffec);
        map.set('Hyper_L', 0xffed);
        map.set('Hyper_R', 0xffee);
        map.set('ISO_Lock', 0xfe01);
        map.set('ISO_Level2_Latch', 0xfe02);
        map.set('ISO_Level3_Shift', 0xfe03);
        map.set('ISO_Level3_Latch', 0xfe04);
        map.set('ISO_Level3_Lock', 0xfe05);
        map.set('ISO_Level5_Shift', 0xfe11);
        map.set('ISO_Level5_Latch', 0xfe12);
        map.set('ISO_Level5_Lock', 0xfe13);
        map.set('ISO_Group_Shift', 0xff7e);
        map.set('ISO_Group_Latch', 0xfe06);
        map.set('ISO_Group_Lock', 0xfe07);
        map.set('ISO_Next_Group', 0xfe08);
        map.set('ISO_Next_Group_Lock', 0xfe09);
        map.set('ISO_Prev_Group', 0xfe0a);
        map.set('ISO_Prev_Group_Lock', 0xfe0b);
        map.set('ISO_First_Group', 0xfe0c);
        map.set('ISO_First_Group_Lock', 0xfe0d);
        map.set('ISO_Last_Group', 0xfe0e);
        map.set('ISO_Last_Group_Lock', 0xfe0f);
        map.set('ISO_Left_Tab', 0xfe20);
        map.set('ISO_Move_Line_Up', 0xfe21);
        map.set('ISO_Move_Line_Down', 0xfe22);
        map.set('ISO_Partial_Line_Up', 0xfe23);
        map.set('ISO_Partial_Line_Down', 0xfe24);
        map.set('ISO_Partial_Space_Left', 0xfe25);
        map.set('ISO_Partial_Space_Right', 0xfe26);
        map.set('ISO_Set_Margin_Left', 0xfe27);
        map.set('ISO_Set_Margin_Right', 0xfe28);
        map.set('ISO_Release_Margin_Left', 0xfe29);
        map.set('ISO_Release_Margin_Right', 0xfe2a);
        map.set('ISO_Release_Both_Margins', 0xfe2b);
        map.set('ISO_Fast_Cursor_Left', 0xfe2c);
        map.set('ISO_Fast_Cursor_Right', 0xfe2d);
        map.set('ISO_Fast_Cursor_Up', 0xfe2e);
        map.set('ISO_Fast_Cursor_Down', 0xfe2f);
        map.set('ISO_Continuous_Underline', 0xfe30);
        map.set('ISO_Discontinuous_Underline', 0xfe31);
        map.set('ISO_Emphasize', 0xfe32);
        map.set('ISO_Center_Object', 0xfe33);
        map.set('ISO_Enter', 0xfe34);
        map.set('dead_grave', 0xfe50);
        map.set('dead_acute', 0xfe51);
        map.set('dead_circumflex', 0xfe52);
        map.set('dead_tilde', 0xfe53);
        map.set('dead_perispomeni', 0xfe53);
        map.set('dead_macron', 0xfe54);
        map.set('dead_breve', 0xfe55);
        map.set('dead_abovedot', 0xfe56);
        map.set('dead_diaeresis', 0xfe57);
        map.set('dead_abovering', 0xfe58);
        map.set('dead_doubleacute', 0xfe59);
        map.set('dead_caron', 0xfe5a);
        map.set('dead_cedilla', 0xfe5b);
        map.set('dead_ogonek', 0xfe5c);
        map.set('dead_iota', 0xfe5d);
        map.set('dead_voiced_sound', 0xfe5e);
        map.set('dead_semivoiced_sound', 0xfe5f);
        map.set('dead_belowdot', 0xfe60);
        map.set('dead_hook', 0xfe61);
        map.set('dead_horn', 0xfe62);
        map.set('dead_stroke', 0xfe63);
        map.set('dead_abovecomma', 0xfe64);
        map.set('dead_psili', 0xfe64);
        map.set('dead_abovereversedcomma', 0xfe65);
        map.set('dead_dasia', 0xfe65);
        map.set('dead_doublegrave', 0xfe66);
        map.set('dead_belowring', 0xfe67);
        map.set('dead_belowmacron', 0xfe68);
        map.set('dead_belowcircumflex', 0xfe69);
        map.set('dead_belowtilde', 0xfe6a);
        map.set('dead_belowbreve', 0xfe6b);
        map.set('dead_belowdiaeresis', 0xfe6c);
        map.set('dead_invertedbreve', 0xfe6d);
        map.set('dead_belowcomma', 0xfe6e);
        map.set('dead_currency', 0xfe6f);
        map.set('dead_lowline', 0xfe90);
        map.set('dead_aboveverticalline', 0xfe91);
        map.set('dead_belowverticalline', 0xfe92);
        map.set('dead_longsolidusoverlay', 0xfe93);
        map.set('dead_a', 0xfe80);
        map.set('dead_A', 0xfe81);
        map.set('dead_e', 0xfe82);
        map.set('dead_E', 0xfe83);
        map.set('dead_i', 0xfe84);
        map.set('dead_I', 0xfe85);
        map.set('dead_o', 0xfe86);
        map.set('dead_O', 0xfe87);
        map.set('dead_u', 0xfe88);
        map.set('dead_U', 0xfe89);
        map.set('dead_small_schwa', 0xfe8a);
        map.set('dead_capital_schwa', 0xfe8b);
        map.set('dead_greek', 0xfe8c);
        map.set('First_Virtual_Screen', 0xfed0);
        map.set('Prev_Virtual_Screen', 0xfed1);
        map.set('Next_Virtual_Screen', 0xfed2);
        map.set('Last_Virtual_Screen', 0xfed4);
        map.set('Terminate_Server', 0xfed5);
        map.set('AccessX_Enable', 0xfe70);
        map.set('AccessX_Feedback_Enable', 0xfe71);
        map.set('RepeatKeys_Enable', 0xfe72);
        map.set('SlowKeys_Enable', 0xfe73);
        map.set('BounceKeys_Enable', 0xfe74);
        map.set('StickyKeys_Enable', 0xfe75);
        map.set('MouseKeys_Enable', 0xfe76);
        map.set('MouseKeys_Accel_Enable', 0xfe77);
        map.set('Overlay1_Enable', 0xfe78);
        map.set('Overlay2_Enable', 0xfe79);
        map.set('AudibleBell_Enable', 0xfe7a);
        map.set('Pointer_Left', 0xfee0);
        map.set('Pointer_Right', 0xfee1);
        map.set('Pointer_Up', 0xfee2);
        map.set('Pointer_Down', 0xfee3);
        map.set('Pointer_UpLeft', 0xfee4);
        map.set('Pointer_UpRight', 0xfee5);
        map.set('Pointer_DownLeft', 0xfee6);
        map.set('Pointer_DownRight', 0xfee7);
        map.set('Pointer_Button_Dflt', 0xfee8);
        map.set('Pointer_Button1', 0xfee9);
        map.set('Pointer_Button2', 0xfeea);
        map.set('Pointer_Button3', 0xfeeb);
        map.set('Pointer_Button4', 0xfeec);
        map.set('Pointer_Button5', 0xfeed);
        map.set('Pointer_DblClick_Dflt', 0xfeee);
        map.set('Pointer_DblClick1', 0xfeef);
        map.set('Pointer_DblClick2', 0xfef0);
        map.set('Pointer_DblClick3', 0xfef1);
        map.set('Pointer_DblClick4', 0xfef2);
        map.set('Pointer_DblClick5', 0xfef3);
        map.set('Pointer_Drag_Dflt', 0xfef4);
        map.set('Pointer_Drag1', 0xfef5);
        map.set('Pointer_Drag2', 0xfef6);
        map.set('Pointer_Drag3', 0xfef7);
        map.set('Pointer_Drag4', 0xfef8);
        map.set('Pointer_Drag5', 0xfefd);
        map.set('Pointer_EnableKeys', 0xfef9);
        map.set('Pointer_Accelerate', 0xfefa);
        map.set('Pointer_DfltBtnNext', 0xfefb);
        map.set('Pointer_DfltBtnPrev', 0xfefc);
        map.set('ch', 0xfea0);
        map.set('Ch', 0xfea1);
        map.set('CH', 0xfea2);
        map.set('c_h', 0xfea3);
        map.set('C_h', 0xfea4);
        map.set('C_H', 0xfea5);
        map.set('3270_Duplicate', 0xfd01);
        map.set('3270_FieldMark', 0xfd02);
        map.set('3270_Right2', 0xfd03);
        map.set('3270_Left2', 0xfd04);
        map.set('3270_BackTab', 0xfd05);
        map.set('3270_EraseEOF', 0xfd06);
        map.set('3270_EraseInput', 0xfd07);
        map.set('3270_Reset', 0xfd08);
        map.set('3270_Quit', 0xfd09);
        map.set('3270_PA1', 0xfd0a);
        map.set('3270_PA2', 0xfd0b);
        map.set('3270_PA3', 0xfd0c);
        map.set('3270_Test', 0xfd0d);
        map.set('3270_Attn', 0xfd0e);
        map.set('3270_CursorBlink', 0xfd0f);
        map.set('3270_AltCursor', 0xfd10);
        map.set('3270_KeyClick', 0xfd11);
        map.set('3270_Jump', 0xfd12);
        map.set('3270_Ident', 0xfd13);
        map.set('3270_Rule', 0xfd14);
        map.set('3270_Copy', 0xfd15);
        map.set('3270_Play', 0xfd16);
        map.set('3270_Setup', 0xfd17);
        map.set('3270_Record', 0xfd18);
        map.set('3270_ChangeScreen', 0xfd19);
        map.set('3270_DeleteWord', 0xfd1a);
        map.set('3270_ExSelect', 0xfd1b);
        map.set('3270_CursorSelect', 0xfd1c);
        map.set('3270_PrintScreen', 0xfd1d);
        map.set('3270_Enter', 0xfd1e);
        map.set('space', 0x020);
        map.set('exclam', 0x021);
        map.set('quotedbl', 0x022);
        map.set('numbersign', 0x023);
        map.set('dollar', 0x024);
        map.set('percent', 0x025);
        map.set('ampersand', 0x026);
        map.set('apostrophe', 0x027);
        map.set('quoteright', 0x027);
        map.set('parenleft', 0x028);
        map.set('parenright', 0x029);
        map.set('asterisk', 0x02a);
        map.set('plus', 0x02b);
        map.set('comma', 0x02c);
        map.set('minus', 0x02d);
        map.set('period', 0x02e);
        map.set('slash', 0x02f);
        map.set('0', 0x030);
        map.set('1', 0x031);
        map.set('2', 0x032);
        map.set('3', 0x033);
        map.set('4', 0x034);
        map.set('5', 0x035);
        map.set('6', 0x036);
        map.set('7', 0x037);
        map.set('8', 0x038);
        map.set('9', 0x039);
        map.set('colon', 0x03a);
        map.set('semicolon', 0x03b);
        map.set('less', 0x03c);
        map.set('equal', 0x03d);
        map.set('greater', 0x03e);
        map.set('question', 0x03f);
        map.set('at', 0x040);
        map.set('A', 0x041);
        map.set('B', 0x042);
        map.set('C', 0x043);
        map.set('D', 0x044);
        map.set('E', 0x045);
        map.set('F', 0x046);
        map.set('G', 0x047);
        map.set('H', 0x048);
        map.set('I', 0x049);
        map.set('J', 0x04a);
        map.set('K', 0x04b);
        map.set('L', 0x04c);
        map.set('M', 0x04d);
        map.set('N', 0x04e);
        map.set('O', 0x04f);
        map.set('P', 0x050);
        map.set('Q', 0x051);
        map.set('R', 0x052);
        map.set('S', 0x053);
        map.set('T', 0x054);
        map.set('U', 0x055);
        map.set('V', 0x056);
        map.set('W', 0x057);
        map.set('X', 0x058);
        map.set('Y', 0x059);
        map.set('Z', 0x05a);
        map.set('bracketleft', 0x05b);
        map.set('backslash', 0x05c);
        map.set('bracketright', 0x05d);
        map.set('asciicircum', 0x05e);
        map.set('underscore', 0x05f);
        map.set('grave', 0x060);
        map.set('quoteleft', 0x060);
        map.set('a', 0x061);
        map.set('b', 0x062);
        map.set('c', 0x063);
        map.set('d', 0x064);
        map.set('e', 0x065);
        map.set('f', 0x066);
        map.set('g', 0x067);
        map.set('h', 0x068);
        map.set('i', 0x069);
        map.set('j', 0x06a);
        map.set('k', 0x06b);
        map.set('l', 0x06c);
        map.set('m', 0x06d);
        map.set('n', 0x06e);
        map.set('o', 0x06f);
        map.set('p', 0x070);
        map.set('q', 0x071);
        map.set('r', 0x072);
        map.set('s', 0x073);
        map.set('t', 0x074);
        map.set('u', 0x075);
        map.set('v', 0x076);
        map.set('w', 0x077);
        map.set('x', 0x078);
        map.set('y', 0x079);
        map.set('z', 0x07a);
        map.set('braceleft', 0x07b);
        map.set('bar', 0x07c);
        map.set('braceright', 0x07d);
        map.set('asciitilde', 0x07e);
        map.set('nobreakspace', 0x0a0);
        map.set('exclamdown', 0x0a1);
        map.set('cent', 0x0a2);
        map.set('sterling', 0x0a3);
        map.set('currency', 0x0a4);
        map.set('yen', 0x0a5);
        map.set('brokenbar', 0x0a6);
        map.set('section', 0x0a7);
        map.set('diaeresis', 0x0a8);
        map.set('copyright', 0x0a9);
        map.set('ordfeminine', 0x0aa);
        map.set('guillemotleft', 0x0ab);
        map.set('notsign', 0x0ac);
        map.set('hyphen', 0x0ad);
        map.set('registered', 0x0ae);
        map.set('macron', 0x0af);
        map.set('degree', 0x0b0);
        map.set('plusminus', 0x0b1);
        map.set('twosuperior', 0x0b2);
        map.set('threesuperior', 0x0b3);
        map.set('acute', 0x0b4);
        map.set('mu', 0x0b5);
        map.set('paragraph', 0x0b6);
        map.set('periodcentered', 0x0b7);
        map.set('cedilla', 0x0b8);
        map.set('onesuperior', 0x0b9);
        map.set('masculine', 0x0ba);
        map.set('guillemotright', 0x0bb);
        map.set('onequarter', 0x0bc);
        map.set('onehalf', 0x0bd);
        map.set('threequarters', 0x0be);
        map.set('questiondown', 0x0bf);
        map.set('Agrave', 0x0c0);
        map.set('Aacute', 0x0c1);
        map.set('Acircumflex', 0x0c2);
        map.set('Atilde', 0x0c3);
        map.set('Adiaeresis', 0x0c4);
        map.set('Aring', 0x0c5);
        map.set('AE', 0x0c6);
        map.set('Ccedilla', 0x0c7);
        map.set('Egrave', 0x0c8);
        map.set('Eacute', 0x0c9);
        map.set('Ecircumflex', 0x0ca);
        map.set('Ediaeresis', 0x0cb);
        map.set('Igrave', 0x0cc);
        map.set('Iacute', 0x0cd);
        map.set('Icircumflex', 0x0ce);
        map.set('Idiaeresis', 0x0cf);
        map.set('ETH', 0x0d0);
        map.set('Eth', 0x0d0);
        map.set('Ntilde', 0x0d1);
        map.set('Ograve', 0x0d2);
        map.set('Oacute', 0x0d3);
        map.set('Ocircumflex', 0x0d4);
        map.set('Otilde', 0x0d5);
        map.set('Odiaeresis', 0x0d6);
        map.set('multiply', 0x0d7);
        map.set('Oslash', 0x0d8);
        map.set('Ooblique', 0x0d8);
        map.set('Ugrave', 0x0d9);
        map.set('Uacute', 0x0da);
        map.set('Ucircumflex', 0x0db);
        map.set('Udiaeresis', 0x0dc);
        map.set('Yacute', 0x0dd);
        map.set('THORN', 0x0de);
        map.set('Thorn', 0x0de);
        map.set('ssharp', 0x0df);
        map.set('agrave', 0x0e0);
        map.set('aacute', 0x0e1);
        map.set('acircumflex', 0x0e2);
        map.set('atilde', 0x0e3);
        map.set('adiaeresis', 0x0e4);
        map.set('aring', 0x0e5);
        map.set('ae', 0x0e6);
        map.set('ccedilla', 0x0e7);
        map.set('egrave', 0x0e8);
        map.set('eacute', 0x0e9);
        map.set('ecircumflex', 0x0ea);
        map.set('ediaeresis', 0x0eb);
        map.set('igrave', 0x0ec);
        map.set('iacute', 0x0ed);
        map.set('icircumflex', 0x0ee);
        map.set('idiaeresis', 0x0ef);
        map.set('eth', 0x0f0);
        map.set('ntilde', 0x0f1);
        map.set('ograve', 0x0f2);
        map.set('oacute', 0x0f3);
        map.set('ocircumflex', 0x0f4);
        map.set('otilde', 0x0f5);
        map.set('odiaeresis', 0x0f6);
        map.set('division', 0x0f7);
        map.set('oslash', 0x0f8);
        map.set('ooblique', 0x0f8);
        map.set('ugrave', 0x0f9);
        map.set('uacute', 0x0fa);
        map.set('ucircumflex', 0x0fb);
        map.set('udiaeresis', 0x0fc);
        map.set('yacute', 0x0fd);
        map.set('thorn', 0x0fe);
        map.set('ydiaeresis', 0x0ff);
        map.set('Aogonek', 0x1a1);
        map.set('breve', 0x1a2);
        map.set('Lstroke', 0x1a3);
        map.set('Lcaron', 0x1a5);
        map.set('Sacute', 0x1a6);
        map.set('Scaron', 0x1a9);
        map.set('Scedilla', 0x1aa);
        map.set('Tcaron', 0x1ab);
        map.set('Zacute', 0x1ac);
        map.set('Zcaron', 0x1ae);
        map.set('Zabovedot', 0x1af);
        map.set('aogonek', 0x1b1);
        map.set('ogonek', 0x1b2);
        map.set('lstroke', 0x1b3);
        map.set('lcaron', 0x1b5);
        map.set('sacute', 0x1b6);
        map.set('caron', 0x1b7);
        map.set('scaron', 0x1b9);
        map.set('scedilla', 0x1ba);
        map.set('tcaron', 0x1bb);
        map.set('zacute', 0x1bc);
        map.set('doubleacute', 0x1bd);
        map.set('zcaron', 0x1be);
        map.set('zabovedot', 0x1bf);
        map.set('Racute', 0x1c0);
        map.set('Abreve', 0x1c3);
        map.set('Lacute', 0x1c5);
        map.set('Cacute', 0x1c6);
        map.set('Ccaron', 0x1c8);
        map.set('Eogonek', 0x1ca);
        map.set('Ecaron', 0x1cc);
        map.set('Dcaron', 0x1cf);
        map.set('Dstroke', 0x1d0);
        map.set('Nacute', 0x1d1);
        map.set('Ncaron', 0x1d2);
        map.set('Odoubleacute', 0x1d5);
        map.set('Rcaron', 0x1d8);
        map.set('Uring', 0x1d9);
        map.set('Udoubleacute', 0x1db);
        map.set('Tcedilla', 0x1de);
        map.set('racute', 0x1e0);
        map.set('abreve', 0x1e3);
        map.set('lacute', 0x1e5);
        map.set('cacute', 0x1e6);
        map.set('ccaron', 0x1e8);
        map.set('eogonek', 0x1ea);
        map.set('ecaron', 0x1ec);
        map.set('dcaron', 0x1ef);
        map.set('dstroke', 0x1f0);
        map.set('nacute', 0x1f1);
        map.set('ncaron', 0x1f2);
        map.set('odoubleacute', 0x1f5);
        map.set('rcaron', 0x1f8);
        map.set('uring', 0x1f9);
        map.set('udoubleacute', 0x1fb);
        map.set('tcedilla', 0x1fe);
        map.set('abovedot', 0x1ff);
        map.set('Hstroke', 0x2a1);
        map.set('Hcircumflex', 0x2a6);
        map.set('Iabovedot', 0x2a9);
        map.set('Gbreve', 0x2ab);
        map.set('Jcircumflex', 0x2ac);
        map.set('hstroke', 0x2b1);
        map.set('hcircumflex', 0x2b6);
        map.set('idotless', 0x2b9);
        map.set('gbreve', 0x2bb);
        map.set('jcircumflex', 0x2bc);
        map.set('Cabovedot', 0x2c5);
        map.set('Ccircumflex', 0x2c6);
        map.set('Gabovedot', 0x2d5);
        map.set('Gcircumflex', 0x2d8);
        map.set('Ubreve', 0x2dd);
        map.set('Scircumflex', 0x2de);
        map.set('cabovedot', 0x2e5);
        map.set('ccircumflex', 0x2e6);
        map.set('gabovedot', 0x2f5);
        map.set('gcircumflex', 0x2f8);
        map.set('ubreve', 0x2fd);
        map.set('scircumflex', 0x2fe);
        map.set('kra', 0x3a2);
        map.set('kappa', 0x3a2);
        map.set('Rcedilla', 0x3a3);
        map.set('Itilde', 0x3a5);
        map.set('Lcedilla', 0x3a6);
        map.set('Emacron', 0x3aa);
        map.set('Gcedilla', 0x3ab);
        map.set('Tslash', 0x3ac);
        map.set('rcedilla', 0x3b3);
        map.set('itilde', 0x3b5);
        map.set('lcedilla', 0x3b6);
        map.set('emacron', 0x3ba);
        map.set('gcedilla', 0x3bb);
        map.set('tslash', 0x3bc);
        map.set('ENG', 0x3bd);
        map.set('eng', 0x3bf);
        map.set('Amacron', 0x3c0);
        map.set('Iogonek', 0x3c7);
        map.set('Eabovedot', 0x3cc);
        map.set('Imacron', 0x3cf);
        map.set('Ncedilla', 0x3d1);
        map.set('Omacron', 0x3d2);
        map.set('Kcedilla', 0x3d3);
        map.set('Uogonek', 0x3d9);
        map.set('Utilde', 0x3dd);
        map.set('Umacron', 0x3de);
        map.set('amacron', 0x3e0);
        map.set('iogonek', 0x3e7);
        map.set('eabovedot', 0x3ec);
        map.set('imacron', 0x3ef);
        map.set('ncedilla', 0x3f1);
        map.set('omacron', 0x3f2);
        map.set('kcedilla', 0x3f3);
        map.set('uogonek', 0x3f9);
        map.set('utilde', 0x3fd);
        map.set('umacron', 0x3fe);
        map.set('Wcircumflex', 0x1000174);
        map.set('wcircumflex', 0x1000175);
        map.set('Ycircumflex', 0x1000176);
        map.set('ycircumflex', 0x1000177);
        map.set('Babovedot', 0x1001e02);
        map.set('babovedot', 0x1001e03);
        map.set('Dabovedot', 0x1001e0a);
        map.set('dabovedot', 0x1001e0b);
        map.set('Fabovedot', 0x1001e1e);
        map.set('fabovedot', 0x1001e1f);
        map.set('Mabovedot', 0x1001e40);
        map.set('mabovedot', 0x1001e41);
        map.set('Pabovedot', 0x1001e56);
        map.set('pabovedot', 0x1001e57);
        map.set('Sabovedot', 0x1001e60);
        map.set('sabovedot', 0x1001e61);
        map.set('Tabovedot', 0x1001e6a);
        map.set('tabovedot', 0x1001e6b);
        map.set('Wgrave', 0x1001e80);
        map.set('wgrave', 0x1001e81);
        map.set('Wacute', 0x1001e82);
        map.set('wacute', 0x1001e83);
        map.set('Wdiaeresis', 0x1001e84);
        map.set('wdiaeresis', 0x1001e85);
        map.set('Ygrave', 0x1001ef2);
        map.set('ygrave', 0x1001ef3);
        map.set('OE', 0x13bc);
        map.set('oe', 0x13bd);
        map.set('Ydiaeresis', 0x13be);
        map.set('overline', 0x47e);
        map.set('kana_fullstop', 0x4a1);
        map.set('kana_openingbracket', 0x4a2);
        map.set('kana_closingbracket', 0x4a3);
        map.set('kana_comma', 0x4a4);
        map.set('kana_conjunctive', 0x4a5);
        map.set('kana_middledot', 0x4a5);
        map.set('kana_WO', 0x4a6);
        map.set('kana_a', 0x4a7);
        map.set('kana_i', 0x4a8);
        map.set('kana_u', 0x4a9);
        map.set('kana_e', 0x4aa);
        map.set('kana_o', 0x4ab);
        map.set('kana_ya', 0x4ac);
        map.set('kana_yu', 0x4ad);
        map.set('kana_yo', 0x4ae);
        map.set('kana_tsu', 0x4af);
        map.set('kana_tu', 0x4af);
        map.set('prolongedsound', 0x4b0);
        map.set('kana_A', 0x4b1);
        map.set('kana_I', 0x4b2);
        map.set('kana_U', 0x4b3);
        map.set('kana_E', 0x4b4);
        map.set('kana_O', 0x4b5);
        map.set('kana_KA', 0x4b6);
        map.set('kana_KI', 0x4b7);
        map.set('kana_KU', 0x4b8);
        map.set('kana_KE', 0x4b9);
        map.set('kana_KO', 0x4ba);
        map.set('kana_SA', 0x4bb);
        map.set('kana_SHI', 0x4bc);
        map.set('kana_SU', 0x4bd);
        map.set('kana_SE', 0x4be);
        map.set('kana_SO', 0x4bf);
        map.set('kana_TA', 0x4c0);
        map.set('kana_CHI', 0x4c1);
        map.set('kana_TI', 0x4c1);
        map.set('kana_TSU', 0x4c2);
        map.set('kana_TU', 0x4c2);
        map.set('kana_TE', 0x4c3);
        map.set('kana_TO', 0x4c4);
        map.set('kana_NA', 0x4c5);
        map.set('kana_NI', 0x4c6);
        map.set('kana_NU', 0x4c7);
        map.set('kana_NE', 0x4c8);
        map.set('kana_NO', 0x4c9);
        map.set('kana_HA', 0x4ca);
        map.set('kana_HI', 0x4cb);
        map.set('kana_FU', 0x4cc);
        map.set('kana_HU', 0x4cc);
        map.set('kana_HE', 0x4cd);
        map.set('kana_HO', 0x4ce);
        map.set('kana_MA', 0x4cf);
        map.set('kana_MI', 0x4d0);
        map.set('kana_MU', 0x4d1);
        map.set('kana_ME', 0x4d2);
        map.set('kana_MO', 0x4d3);
        map.set('kana_YA', 0x4d4);
        map.set('kana_YU', 0x4d5);
        map.set('kana_YO', 0x4d6);
        map.set('kana_RA', 0x4d7);
        map.set('kana_RI', 0x4d8);
        map.set('kana_RU', 0x4d9);
        map.set('kana_RE', 0x4da);
        map.set('kana_RO', 0x4db);
        map.set('kana_WA', 0x4dc);
        map.set('kana_N', 0x4dd);
        map.set('voicedsound', 0x4de);
        map.set('semivoicedsound', 0x4df);
        map.set('kana_switch', 0xff7e);
        map.set('Farsi_0', 0x10006f0);
        map.set('Farsi_1', 0x10006f1);
        map.set('Farsi_2', 0x10006f2);
        map.set('Farsi_3', 0x10006f3);
        map.set('Farsi_4', 0x10006f4);
        map.set('Farsi_5', 0x10006f5);
        map.set('Farsi_6', 0x10006f6);
        map.set('Farsi_7', 0x10006f7);
        map.set('Farsi_8', 0x10006f8);
        map.set('Farsi_9', 0x10006f9);
        map.set('Arabic_percent', 0x100066a);
        map.set('Arabic_superscript_alef', 0x1000670);
        map.set('Arabic_tteh', 0x1000679);
        map.set('Arabic_peh', 0x100067e);
        map.set('Arabic_tcheh', 0x1000686);
        map.set('Arabic_ddal', 0x1000688);
        map.set('Arabic_rreh', 0x1000691);
        map.set('Arabic_comma', 0x5ac);
        map.set('Arabic_fullstop', 0x10006d4);
        map.set('Arabic_0', 0x1000660);
        map.set('Arabic_1', 0x1000661);
        map.set('Arabic_2', 0x1000662);
        map.set('Arabic_3', 0x1000663);
        map.set('Arabic_4', 0x1000664);
        map.set('Arabic_5', 0x1000665);
        map.set('Arabic_6', 0x1000666);
        map.set('Arabic_7', 0x1000667);
        map.set('Arabic_8', 0x1000668);
        map.set('Arabic_9', 0x1000669);
        map.set('Arabic_semicolon', 0x5bb);
        map.set('Arabic_question_mark', 0x5bf);
        map.set('Arabic_hamza', 0x5c1);
        map.set('Arabic_maddaonalef', 0x5c2);
        map.set('Arabic_hamzaonalef', 0x5c3);
        map.set('Arabic_hamzaonwaw', 0x5c4);
        map.set('Arabic_hamzaunderalef', 0x5c5);
        map.set('Arabic_hamzaonyeh', 0x5c6);
        map.set('Arabic_alef', 0x5c7);
        map.set('Arabic_beh', 0x5c8);
        map.set('Arabic_tehmarbuta', 0x5c9);
        map.set('Arabic_teh', 0x5ca);
        map.set('Arabic_theh', 0x5cb);
        map.set('Arabic_jeem', 0x5cc);
        map.set('Arabic_hah', 0x5cd);
        map.set('Arabic_khah', 0x5ce);
        map.set('Arabic_dal', 0x5cf);
        map.set('Arabic_thal', 0x5d0);
        map.set('Arabic_ra', 0x5d1);
        map.set('Arabic_zain', 0x5d2);
        map.set('Arabic_seen', 0x5d3);
        map.set('Arabic_sheen', 0x5d4);
        map.set('Arabic_sad', 0x5d5);
        map.set('Arabic_dad', 0x5d6);
        map.set('Arabic_tah', 0x5d7);
        map.set('Arabic_zah', 0x5d8);
        map.set('Arabic_ain', 0x5d9);
        map.set('Arabic_ghain', 0x5da);
        map.set('Arabic_tatweel', 0x5e0);
        map.set('Arabic_feh', 0x5e1);
        map.set('Arabic_qaf', 0x5e2);
        map.set('Arabic_kaf', 0x5e3);
        map.set('Arabic_lam', 0x5e4);
        map.set('Arabic_meem', 0x5e5);
        map.set('Arabic_noon', 0x5e6);
        map.set('Arabic_ha', 0x5e7);
        map.set('Arabic_heh', 0x5e7);
        map.set('Arabic_waw', 0x5e8);
        map.set('Arabic_alefmaksura', 0x5e9);
        map.set('Arabic_yeh', 0x5ea);
        map.set('Arabic_fathatan', 0x5eb);
        map.set('Arabic_dammatan', 0x5ec);
        map.set('Arabic_kasratan', 0x5ed);
        map.set('Arabic_fatha', 0x5ee);
        map.set('Arabic_damma', 0x5ef);
        map.set('Arabic_kasra', 0x5f0);
        map.set('Arabic_shadda', 0x5f1);
        map.set('Arabic_sukun', 0x5f2);
        map.set('Arabic_madda_above', 0x1000653);
        map.set('Arabic_hamza_above', 0x1000654);
        map.set('Arabic_hamza_below', 0x1000655);
        map.set('Arabic_jeh', 0x1000698);
        map.set('Arabic_veh', 0x10006a4);
        map.set('Arabic_keheh', 0x10006a9);
        map.set('Arabic_gaf', 0x10006af);
        map.set('Arabic_noon_ghunna', 0x10006ba);
        map.set('Arabic_heh_doachashmee', 0x10006be);
        map.set('Farsi_yeh', 0x10006cc);
        map.set('Arabic_farsi_yeh', 0x10006cc);
        map.set('Arabic_yeh_baree', 0x10006d2);
        map.set('Arabic_heh_goal', 0x10006c1);
        map.set('Arabic_switch', 0xff7e);
        map.set('Cyrillic_GHE_bar', 0x1000492);
        map.set('Cyrillic_ghe_bar', 0x1000493);
        map.set('Cyrillic_ZHE_descender', 0x1000496);
        map.set('Cyrillic_zhe_descender', 0x1000497);
        map.set('Cyrillic_KA_descender', 0x100049a);
        map.set('Cyrillic_ka_descender', 0x100049b);
        map.set('Cyrillic_KA_vertstroke', 0x100049c);
        map.set('Cyrillic_ka_vertstroke', 0x100049d);
        map.set('Cyrillic_EN_descender', 0x10004a2);
        map.set('Cyrillic_en_descender', 0x10004a3);
        map.set('Cyrillic_U_straight', 0x10004ae);
        map.set('Cyrillic_u_straight', 0x10004af);
        map.set('Cyrillic_U_straight_bar', 0x10004b0);
        map.set('Cyrillic_u_straight_bar', 0x10004b1);
        map.set('Cyrillic_HA_descender', 0x10004b2);
        map.set('Cyrillic_ha_descender', 0x10004b3);
        map.set('Cyrillic_CHE_descender', 0x10004b6);
        map.set('Cyrillic_che_descender', 0x10004b7);
        map.set('Cyrillic_CHE_vertstroke', 0x10004b8);
        map.set('Cyrillic_che_vertstroke', 0x10004b9);
        map.set('Cyrillic_SHHA', 0x10004ba);
        map.set('Cyrillic_shha', 0x10004bb);
        map.set('Cyrillic_SCHWA', 0x10004d8);
        map.set('Cyrillic_schwa', 0x10004d9);
        map.set('Cyrillic_I_macron', 0x10004e2);
        map.set('Cyrillic_i_macron', 0x10004e3);
        map.set('Cyrillic_O_bar', 0x10004e8);
        map.set('Cyrillic_o_bar', 0x10004e9);
        map.set('Cyrillic_U_macron', 0x10004ee);
        map.set('Cyrillic_u_macron', 0x10004ef);
        map.set('Serbian_dje', 0x6a1);
        map.set('Macedonia_gje', 0x6a2);
        map.set('Cyrillic_io', 0x6a3);
        map.set('Ukrainian_ie', 0x6a4);
        map.set('Ukranian_je', 0x6a4);
        map.set('Macedonia_dse', 0x6a5);
        map.set('Ukrainian_i', 0x6a6);
        map.set('Ukranian_i', 0x6a6);
        map.set('Ukrainian_yi', 0x6a7);
        map.set('Ukranian_yi', 0x6a7);
        map.set('Cyrillic_je', 0x6a8);
        map.set('Serbian_je', 0x6a8);
        map.set('Cyrillic_lje', 0x6a9);
        map.set('Serbian_lje', 0x6a9);
        map.set('Cyrillic_nje', 0x6aa);
        map.set('Serbian_nje', 0x6aa);
        map.set('Serbian_tshe', 0x6ab);
        map.set('Macedonia_kje', 0x6ac);
        map.set('Ukrainian_ghe_with_upturn', 0x6ad);
        map.set('Byelorussian_shortu', 0x6ae);
        map.set('Cyrillic_dzhe', 0x6af);
        map.set('Serbian_dze', 0x6af);
        map.set('numerosign', 0x6b0);
        map.set('Serbian_DJE', 0x6b1);
        map.set('Macedonia_GJE', 0x6b2);
        map.set('Cyrillic_IO', 0x6b3);
        map.set('Ukrainian_IE', 0x6b4);
        map.set('Ukranian_JE', 0x6b4);
        map.set('Macedonia_DSE', 0x6b5);
        map.set('Ukrainian_I', 0x6b6);
        map.set('Ukranian_I', 0x6b6);
        map.set('Ukrainian_YI', 0x6b7);
        map.set('Ukranian_YI', 0x6b7);
        map.set('Cyrillic_JE', 0x6b8);
        map.set('Serbian_JE', 0x6b8);
        map.set('Cyrillic_LJE', 0x6b9);
        map.set('Serbian_LJE', 0x6b9);
        map.set('Cyrillic_NJE', 0x6ba);
        map.set('Serbian_NJE', 0x6ba);
        map.set('Serbian_TSHE', 0x6bb);
        map.set('Macedonia_KJE', 0x6bc);
        map.set('Ukrainian_GHE_WITH_UPTURN', 0x6bd);
        map.set('Byelorussian_SHORTU', 0x6be);
        map.set('Cyrillic_DZHE', 0x6bf);
        map.set('Serbian_DZE', 0x6bf);
        map.set('Cyrillic_yu', 0x6c0);
        map.set('Cyrillic_a', 0x6c1);
        map.set('Cyrillic_be', 0x6c2);
        map.set('Cyrillic_tse', 0x6c3);
        map.set('Cyrillic_de', 0x6c4);
        map.set('Cyrillic_ie', 0x6c5);
        map.set('Cyrillic_ef', 0x6c6);
        map.set('Cyrillic_ghe', 0x6c7);
        map.set('Cyrillic_ha', 0x6c8);
        map.set('Cyrillic_i', 0x6c9);
        map.set('Cyrillic_shorti', 0x6ca);
        map.set('Cyrillic_ka', 0x6cb);
        map.set('Cyrillic_el', 0x6cc);
        map.set('Cyrillic_em', 0x6cd);
        map.set('Cyrillic_en', 0x6ce);
        map.set('Cyrillic_o', 0x6cf);
        map.set('Cyrillic_pe', 0x6d0);
        map.set('Cyrillic_ya', 0x6d1);
        map.set('Cyrillic_er', 0x6d2);
        map.set('Cyrillic_es', 0x6d3);
        map.set('Cyrillic_te', 0x6d4);
        map.set('Cyrillic_u', 0x6d5);
        map.set('Cyrillic_zhe', 0x6d6);
        map.set('Cyrillic_ve', 0x6d7);
        map.set('Cyrillic_softsign', 0x6d8);
        map.set('Cyrillic_yeru', 0x6d9);
        map.set('Cyrillic_ze', 0x6da);
        map.set('Cyrillic_sha', 0x6db);
        map.set('Cyrillic_e', 0x6dc);
        map.set('Cyrillic_shcha', 0x6dd);
        map.set('Cyrillic_che', 0x6de);
        map.set('Cyrillic_hardsign', 0x6df);
        map.set('Cyrillic_YU', 0x6e0);
        map.set('Cyrillic_A', 0x6e1);
        map.set('Cyrillic_BE', 0x6e2);
        map.set('Cyrillic_TSE', 0x6e3);
        map.set('Cyrillic_DE', 0x6e4);
        map.set('Cyrillic_IE', 0x6e5);
        map.set('Cyrillic_EF', 0x6e6);
        map.set('Cyrillic_GHE', 0x6e7);
        map.set('Cyrillic_HA', 0x6e8);
        map.set('Cyrillic_I', 0x6e9);
        map.set('Cyrillic_SHORTI', 0x6ea);
        map.set('Cyrillic_KA', 0x6eb);
        map.set('Cyrillic_EL', 0x6ec);
        map.set('Cyrillic_EM', 0x6ed);
        map.set('Cyrillic_EN', 0x6ee);
        map.set('Cyrillic_O', 0x6ef);
        map.set('Cyrillic_PE', 0x6f0);
        map.set('Cyrillic_YA', 0x6f1);
        map.set('Cyrillic_ER', 0x6f2);
        map.set('Cyrillic_ES', 0x6f3);
        map.set('Cyrillic_TE', 0x6f4);
        map.set('Cyrillic_U', 0x6f5);
        map.set('Cyrillic_ZHE', 0x6f6);
        map.set('Cyrillic_VE', 0x6f7);
        map.set('Cyrillic_SOFTSIGN', 0x6f8);
        map.set('Cyrillic_YERU', 0x6f9);
        map.set('Cyrillic_ZE', 0x6fa);
        map.set('Cyrillic_SHA', 0x6fb);
        map.set('Cyrillic_E', 0x6fc);
        map.set('Cyrillic_SHCHA', 0x6fd);
        map.set('Cyrillic_CHE', 0x6fe);
        map.set('Cyrillic_HARDSIGN', 0x6ff);
        map.set('Greek_ALPHAaccent', 0x7a1);
        map.set('Greek_EPSILONaccent', 0x7a2);
        map.set('Greek_ETAaccent', 0x7a3);
        map.set('Greek_IOTAaccent', 0x7a4);
        map.set('Greek_IOTAdieresis', 0x7a5);
        map.set('Greek_IOTAdiaeresis', 0x7a5);
        map.set('Greek_OMICRONaccent', 0x7a7);
        map.set('Greek_UPSILONaccent', 0x7a8);
        map.set('Greek_UPSILONdieresis', 0x7a9);
        map.set('Greek_OMEGAaccent', 0x7ab);
        map.set('Greek_accentdieresis', 0x7ae);
        map.set('Greek_horizbar', 0x7af);
        map.set('Greek_alphaaccent', 0x7b1);
        map.set('Greek_epsilonaccent', 0x7b2);
        map.set('Greek_etaaccent', 0x7b3);
        map.set('Greek_iotaaccent', 0x7b4);
        map.set('Greek_iotadieresis', 0x7b5);
        map.set('Greek_iotaaccentdieresis', 0x7b6);
        map.set('Greek_omicronaccent', 0x7b7);
        map.set('Greek_upsilonaccent', 0x7b8);
        map.set('Greek_upsilondieresis', 0x7b9);
        map.set('Greek_upsilonaccentdieresis', 0x7ba);
        map.set('Greek_omegaaccent', 0x7bb);
        map.set('Greek_ALPHA', 0x7c1);
        map.set('Greek_BETA', 0x7c2);
        map.set('Greek_GAMMA', 0x7c3);
        map.set('Greek_DELTA', 0x7c4);
        map.set('Greek_EPSILON', 0x7c5);
        map.set('Greek_ZETA', 0x7c6);
        map.set('Greek_ETA', 0x7c7);
        map.set('Greek_THETA', 0x7c8);
        map.set('Greek_IOTA', 0x7c9);
        map.set('Greek_KAPPA', 0x7ca);
        map.set('Greek_LAMDA', 0x7cb);
        map.set('Greek_LAMBDA', 0x7cb);
        map.set('Greek_MU', 0x7cc);
        map.set('Greek_NU', 0x7cd);
        map.set('Greek_XI', 0x7ce);
        map.set('Greek_OMICRON', 0x7cf);
        map.set('Greek_PI', 0x7d0);
        map.set('Greek_RHO', 0x7d1);
        map.set('Greek_SIGMA', 0x7d2);
        map.set('Greek_TAU', 0x7d4);
        map.set('Greek_UPSILON', 0x7d5);
        map.set('Greek_PHI', 0x7d6);
        map.set('Greek_CHI', 0x7d7);
        map.set('Greek_PSI', 0x7d8);
        map.set('Greek_OMEGA', 0x7d9);
        map.set('Greek_alpha', 0x7e1);
        map.set('Greek_beta', 0x7e2);
        map.set('Greek_gamma', 0x7e3);
        map.set('Greek_delta', 0x7e4);
        map.set('Greek_epsilon', 0x7e5);
        map.set('Greek_zeta', 0x7e6);
        map.set('Greek_eta', 0x7e7);
        map.set('Greek_theta', 0x7e8);
        map.set('Greek_iota', 0x7e9);
        map.set('Greek_kappa', 0x7ea);
        map.set('Greek_lamda', 0x7eb);
        map.set('Greek_lambda', 0x7eb);
        map.set('Greek_mu', 0x7ec);
        map.set('Greek_nu', 0x7ed);
        map.set('Greek_xi', 0x7ee);
        map.set('Greek_omicron', 0x7ef);
        map.set('Greek_pi', 0x7f0);
        map.set('Greek_rho', 0x7f1);
        map.set('Greek_sigma', 0x7f2);
        map.set('Greek_finalsmallsigma', 0x7f3);
        map.set('Greek_tau', 0x7f4);
        map.set('Greek_upsilon', 0x7f5);
        map.set('Greek_phi', 0x7f6);
        map.set('Greek_chi', 0x7f7);
        map.set('Greek_psi', 0x7f8);
        map.set('Greek_omega', 0x7f9);
        map.set('Greek_switch', 0xff7e);
        map.set('leftradical', 0x8a1);
        map.set('topleftradical', 0x8a2);
        map.set('horizconnector', 0x8a3);
        map.set('topintegral', 0x8a4);
        map.set('botintegral', 0x8a5);
        map.set('vertconnector', 0x8a6);
        map.set('topleftsqbracket', 0x8a7);
        map.set('botleftsqbracket', 0x8a8);
        map.set('toprightsqbracket', 0x8a9);
        map.set('botrightsqbracket', 0x8aa);
        map.set('topleftparens', 0x8ab);
        map.set('botleftparens', 0x8ac);
        map.set('toprightparens', 0x8ad);
        map.set('botrightparens', 0x8ae);
        map.set('leftmiddlecurlybrace', 0x8af);
        map.set('rightmiddlecurlybrace', 0x8b0);
        map.set('topleftsummation', 0x8b1);
        map.set('botleftsummation', 0x8b2);
        map.set('topvertsummationconnector', 0x8b3);
        map.set('botvertsummationconnector', 0x8b4);
        map.set('toprightsummation', 0x8b5);
        map.set('botrightsummation', 0x8b6);
        map.set('rightmiddlesummation', 0x8b7);
        map.set('lessthanequal', 0x8bc);
        map.set('notequal', 0x8bd);
        map.set('greaterthanequal', 0x8be);
        map.set('integral', 0x8bf);
        map.set('therefore', 0x8c0);
        map.set('variation', 0x8c1);
        map.set('infinity', 0x8c2);
        map.set('nabla', 0x8c5);
        map.set('approximate', 0x8c8);
        map.set('similarequal', 0x8c9);
        map.set('ifonlyif', 0x8cd);
        map.set('implies', 0x8ce);
        map.set('identical', 0x8cf);
        map.set('radical', 0x8d6);
        map.set('includedin', 0x8da);
        map.set('includes', 0x8db);
        map.set('intersection', 0x8dc);
        map.set('union', 0x8dd);
        map.set('logicaland', 0x8de);
        map.set('logicalor', 0x8df);
        map.set('partialderivative', 0x8ef);
        map.set('function', 0x8f6);
        map.set('leftarrow', 0x8fb);
        map.set('uparrow', 0x8fc);
        map.set('rightarrow', 0x8fd);
        map.set('downarrow', 0x8fe);
        map.set('blank', 0x9df);
        map.set('soliddiamond', 0x9e0);
        map.set('checkerboard', 0x9e1);
        map.set('ht', 0x9e2);
        map.set('ff', 0x9e3);
        map.set('cr', 0x9e4);
        map.set('lf', 0x9e5);
        map.set('nl', 0x9e8);
        map.set('vt', 0x9e9);
        map.set('lowrightcorner', 0x9ea);
        map.set('uprightcorner', 0x9eb);
        map.set('upleftcorner', 0x9ec);
        map.set('lowleftcorner', 0x9ed);
        map.set('crossinglines', 0x9ee);
        map.set('horizlinescan1', 0x9ef);
        map.set('horizlinescan3', 0x9f0);
        map.set('horizlinescan5', 0x9f1);
        map.set('horizlinescan7', 0x9f2);
        map.set('horizlinescan9', 0x9f3);
        map.set('leftt', 0x9f4);
        map.set('rightt', 0x9f5);
        map.set('bott', 0x9f6);
        map.set('topt', 0x9f7);
        map.set('vertbar', 0x9f8);
        map.set('emspace', 0xaa1);
        map.set('enspace', 0xaa2);
        map.set('em3space', 0xaa3);
        map.set('em4space', 0xaa4);
        map.set('digitspace', 0xaa5);
        map.set('punctspace', 0xaa6);
        map.set('thinspace', 0xaa7);
        map.set('hairspace', 0xaa8);
        map.set('emdash', 0xaa9);
        map.set('endash', 0xaaa);
        map.set('signifblank', 0xaac);
        map.set('ellipsis', 0xaae);
        map.set('doubbaselinedot', 0xaaf);
        map.set('onethird', 0xab0);
        map.set('twothirds', 0xab1);
        map.set('onefifth', 0xab2);
        map.set('twofifths', 0xab3);
        map.set('threefifths', 0xab4);
        map.set('fourfifths', 0xab5);
        map.set('onesixth', 0xab6);
        map.set('fivesixths', 0xab7);
        map.set('careof', 0xab8);
        map.set('figdash', 0xabb);
        map.set('leftanglebracket', 0xabc);
        map.set('decimalpoint', 0xabd);
        map.set('rightanglebracket', 0xabe);
        map.set('marker', 0xabf);
        map.set('oneeighth', 0xac3);
        map.set('threeeighths', 0xac4);
        map.set('fiveeighths', 0xac5);
        map.set('seveneighths', 0xac6);
        map.set('trademark', 0xac9);
        map.set('signaturemark', 0xaca);
        map.set('trademarkincircle', 0xacb);
        map.set('leftopentriangle', 0xacc);
        map.set('rightopentriangle', 0xacd);
        map.set('emopencircle', 0xace);
        map.set('emopenrectangle', 0xacf);
        map.set('leftsinglequotemark', 0xad0);
        map.set('rightsinglequotemark', 0xad1);
        map.set('leftdoublequotemark', 0xad2);
        map.set('rightdoublequotemark', 0xad3);
        map.set('prescription', 0xad4);
        map.set('permille', 0xad5);
        map.set('minutes', 0xad6);
        map.set('seconds', 0xad7);
        map.set('latincross', 0xad9);
        map.set('hexagram', 0xada);
        map.set('filledrectbullet', 0xadb);
        map.set('filledlefttribullet', 0xadc);
        map.set('filledrighttribullet', 0xadd);
        map.set('emfilledcircle', 0xade);
        map.set('emfilledrect', 0xadf);
        map.set('enopencircbullet', 0xae0);
        map.set('enopensquarebullet', 0xae1);
        map.set('openrectbullet', 0xae2);
        map.set('opentribulletup', 0xae3);
        map.set('opentribulletdown', 0xae4);
        map.set('openstar', 0xae5);
        map.set('enfilledcircbullet', 0xae6);
        map.set('enfilledsqbullet', 0xae7);
        map.set('filledtribulletup', 0xae8);
        map.set('filledtribulletdown', 0xae9);
        map.set('leftpointer', 0xaea);
        map.set('rightpointer', 0xaeb);
        map.set('club', 0xaec);
        map.set('diamond', 0xaed);
        map.set('heart', 0xaee);
        map.set('maltesecross', 0xaf0);
        map.set('dagger', 0xaf1);
        map.set('doubledagger', 0xaf2);
        map.set('checkmark', 0xaf3);
        map.set('ballotcross', 0xaf4);
        map.set('musicalsharp', 0xaf5);
        map.set('musicalflat', 0xaf6);
        map.set('malesymbol', 0xaf7);
        map.set('femalesymbol', 0xaf8);
        map.set('telephone', 0xaf9);
        map.set('telephonerecorder', 0xafa);
        map.set('phonographcopyright', 0xafb);
        map.set('caret', 0xafc);
        map.set('singlelowquotemark', 0xafd);
        map.set('doublelowquotemark', 0xafe);
        map.set('cursor', 0xaff);
        map.set('leftcaret', 0xba3);
        map.set('rightcaret', 0xba6);
        map.set('downcaret', 0xba8);
        map.set('upcaret', 0xba9);
        map.set('overbar', 0xbc0);
        map.set('downtack', 0xbc2);
        map.set('upshoe', 0xbc3);
        map.set('downstile', 0xbc4);
        map.set('underbar', 0xbc6);
        map.set('jot', 0xbca);
        map.set('quad', 0xbcc);
        map.set('uptack', 0xbce);
        map.set('circle', 0xbcf);
        map.set('upstile', 0xbd3);
        map.set('downshoe', 0xbd6);
        map.set('rightshoe', 0xbd8);
        map.set('leftshoe', 0xbda);
        map.set('lefttack', 0xbdc);
        map.set('righttack', 0xbfc);
        map.set('hebrew_doublelowline', 0xcdf);
        map.set('hebrew_aleph', 0xce0);
        map.set('hebrew_bet', 0xce1);
        map.set('hebrew_beth', 0xce1);
        map.set('hebrew_gimel', 0xce2);
        map.set('hebrew_gimmel', 0xce2);
        map.set('hebrew_dalet', 0xce3);
        map.set('hebrew_daleth', 0xce3);
        map.set('hebrew_he', 0xce4);
        map.set('hebrew_waw', 0xce5);
        map.set('hebrew_zain', 0xce6);
        map.set('hebrew_zayin', 0xce6);
        map.set('hebrew_chet', 0xce7);
        map.set('hebrew_het', 0xce7);
        map.set('hebrew_tet', 0xce8);
        map.set('hebrew_teth', 0xce8);
        map.set('hebrew_yod', 0xce9);
        map.set('hebrew_finalkaph', 0xcea);
        map.set('hebrew_kaph', 0xceb);
        map.set('hebrew_lamed', 0xcec);
        map.set('hebrew_finalmem', 0xced);
        map.set('hebrew_mem', 0xcee);
        map.set('hebrew_finalnun', 0xcef);
        map.set('hebrew_nun', 0xcf0);
        map.set('hebrew_samech', 0xcf1);
        map.set('hebrew_samekh', 0xcf1);
        map.set('hebrew_ayin', 0xcf2);
        map.set('hebrew_finalpe', 0xcf3);
        map.set('hebrew_pe', 0xcf4);
        map.set('hebrew_finalzade', 0xcf5);
        map.set('hebrew_finalzadi', 0xcf5);
        map.set('hebrew_zade', 0xcf6);
        map.set('hebrew_zadi', 0xcf6);
        map.set('hebrew_qoph', 0xcf7);
        map.set('hebrew_kuf', 0xcf7);
        map.set('hebrew_resh', 0xcf8);
        map.set('hebrew_shin', 0xcf9);
        map.set('hebrew_taw', 0xcfa);
        map.set('hebrew_taf', 0xcfa);
        map.set('Hebrew_switch', 0xff7e);
        map.set('Thai_kokai', 0xda1);
        map.set('Thai_khokhai', 0xda2);
        map.set('Thai_khokhuat', 0xda3);
        map.set('Thai_khokhwai', 0xda4);
        map.set('Thai_khokhon', 0xda5);
        map.set('Thai_khorakhang', 0xda6);
        map.set('Thai_ngongu', 0xda7);
        map.set('Thai_chochan', 0xda8);
        map.set('Thai_choching', 0xda9);
        map.set('Thai_chochang', 0xdaa);
        map.set('Thai_soso', 0xdab);
        map.set('Thai_chochoe', 0xdac);
        map.set('Thai_yoying', 0xdad);
        map.set('Thai_dochada', 0xdae);
        map.set('Thai_topatak', 0xdaf);
        map.set('Thai_thothan', 0xdb0);
        map.set('Thai_thonangmontho', 0xdb1);
        map.set('Thai_thophuthao', 0xdb2);
        map.set('Thai_nonen', 0xdb3);
        map.set('Thai_dodek', 0xdb4);
        map.set('Thai_totao', 0xdb5);
        map.set('Thai_thothung', 0xdb6);
        map.set('Thai_thothahan', 0xdb7);
        map.set('Thai_thothong', 0xdb8);
        map.set('Thai_nonu', 0xdb9);
        map.set('Thai_bobaimai', 0xdba);
        map.set('Thai_popla', 0xdbb);
        map.set('Thai_phophung', 0xdbc);
        map.set('Thai_fofa', 0xdbd);
        map.set('Thai_phophan', 0xdbe);
        map.set('Thai_fofan', 0xdbf);
        map.set('Thai_phosamphao', 0xdc0);
        map.set('Thai_moma', 0xdc1);
        map.set('Thai_yoyak', 0xdc2);
        map.set('Thai_rorua', 0xdc3);
        map.set('Thai_ru', 0xdc4);
        map.set('Thai_loling', 0xdc5);
        map.set('Thai_lu', 0xdc6);
        map.set('Thai_wowaen', 0xdc7);
        map.set('Thai_sosala', 0xdc8);
        map.set('Thai_sorusi', 0xdc9);
        map.set('Thai_sosua', 0xdca);
        map.set('Thai_hohip', 0xdcb);
        map.set('Thai_lochula', 0xdcc);
        map.set('Thai_oang', 0xdcd);
        map.set('Thai_honokhuk', 0xdce);
        map.set('Thai_paiyannoi', 0xdcf);
        map.set('Thai_saraa', 0xdd0);
        map.set('Thai_maihanakat', 0xdd1);
        map.set('Thai_saraaa', 0xdd2);
        map.set('Thai_saraam', 0xdd3);
        map.set('Thai_sarai', 0xdd4);
        map.set('Thai_saraii', 0xdd5);
        map.set('Thai_saraue', 0xdd6);
        map.set('Thai_sarauee', 0xdd7);
        map.set('Thai_sarau', 0xdd8);
        map.set('Thai_sarauu', 0xdd9);
        map.set('Thai_phinthu', 0xdda);
        map.set('Thai_maihanakat_maitho', 0xdde);
        map.set('Thai_baht', 0xddf);
        map.set('Thai_sarae', 0xde0);
        map.set('Thai_saraae', 0xde1);
        map.set('Thai_sarao', 0xde2);
        map.set('Thai_saraaimaimuan', 0xde3);
        map.set('Thai_saraaimaimalai', 0xde4);
        map.set('Thai_lakkhangyao', 0xde5);
        map.set('Thai_maiyamok', 0xde6);
        map.set('Thai_maitaikhu', 0xde7);
        map.set('Thai_maiek', 0xde8);
        map.set('Thai_maitho', 0xde9);
        map.set('Thai_maitri', 0xdea);
        map.set('Thai_maichattawa', 0xdeb);
        map.set('Thai_thanthakhat', 0xdec);
        map.set('Thai_nikhahit', 0xded);
        map.set('Thai_leksun', 0xdf0);
        map.set('Thai_leknung', 0xdf1);
        map.set('Thai_leksong', 0xdf2);
        map.set('Thai_leksam', 0xdf3);
        map.set('Thai_leksi', 0xdf4);
        map.set('Thai_lekha', 0xdf5);
        map.set('Thai_lekhok', 0xdf6);
        map.set('Thai_lekchet', 0xdf7);
        map.set('Thai_lekpaet', 0xdf8);
        map.set('Thai_lekkao', 0xdf9);
        map.set('Hangul', 0xff31);
        map.set('Hangul_Start', 0xff32);
        map.set('Hangul_End', 0xff33);
        map.set('Hangul_Hanja', 0xff34);
        map.set('Hangul_Jamo', 0xff35);
        map.set('Hangul_Romaja', 0xff36);
        map.set('Hangul_Codeinput', 0xff37);
        map.set('Hangul_Jeonja', 0xff38);
        map.set('Hangul_Banja', 0xff39);
        map.set('Hangul_PreHanja', 0xff3a);
        map.set('Hangul_PostHanja', 0xff3b);
        map.set('Hangul_SingleCandidate', 0xff3c);
        map.set('Hangul_MultipleCandidate', 0xff3d);
        map.set('Hangul_PreviousCandidate', 0xff3e);
        map.set('Hangul_Special', 0xff3f);
        map.set('Hangul_switch', 0xff7e);
        map.set('Hangul_Kiyeog', 0xea1);
        map.set('Hangul_SsangKiyeog', 0xea2);
        map.set('Hangul_KiyeogSios', 0xea3);
        map.set('Hangul_Nieun', 0xea4);
        map.set('Hangul_NieunJieuj', 0xea5);
        map.set('Hangul_NieunHieuh', 0xea6);
        map.set('Hangul_Dikeud', 0xea7);
        map.set('Hangul_SsangDikeud', 0xea8);
        map.set('Hangul_Rieul', 0xea9);
        map.set('Hangul_RieulKiyeog', 0xeaa);
        map.set('Hangul_RieulMieum', 0xeab);
        map.set('Hangul_RieulPieub', 0xeac);
        map.set('Hangul_RieulSios', 0xead);
        map.set('Hangul_RieulTieut', 0xeae);
        map.set('Hangul_RieulPhieuf', 0xeaf);
        map.set('Hangul_RieulHieuh', 0xeb0);
        map.set('Hangul_Mieum', 0xeb1);
        map.set('Hangul_Pieub', 0xeb2);
        map.set('Hangul_SsangPieub', 0xeb3);
        map.set('Hangul_PieubSios', 0xeb4);
        map.set('Hangul_Sios', 0xeb5);
        map.set('Hangul_SsangSios', 0xeb6);
        map.set('Hangul_Ieung', 0xeb7);
        map.set('Hangul_Jieuj', 0xeb8);
        map.set('Hangul_SsangJieuj', 0xeb9);
        map.set('Hangul_Cieuc', 0xeba);
        map.set('Hangul_Khieuq', 0xebb);
        map.set('Hangul_Tieut', 0xebc);
        map.set('Hangul_Phieuf', 0xebd);
        map.set('Hangul_Hieuh', 0xebe);
        map.set('Hangul_A', 0xebf);
        map.set('Hangul_AE', 0xec0);
        map.set('Hangul_YA', 0xec1);
        map.set('Hangul_YAE', 0xec2);
        map.set('Hangul_EO', 0xec3);
        map.set('Hangul_E', 0xec4);
        map.set('Hangul_YEO', 0xec5);
        map.set('Hangul_YE', 0xec6);
        map.set('Hangul_O', 0xec7);
        map.set('Hangul_WA', 0xec8);
        map.set('Hangul_WAE', 0xec9);
        map.set('Hangul_OE', 0xeca);
        map.set('Hangul_YO', 0xecb);
        map.set('Hangul_U', 0xecc);
        map.set('Hangul_WEO', 0xecd);
        map.set('Hangul_WE', 0xece);
        map.set('Hangul_WI', 0xecf);
        map.set('Hangul_YU', 0xed0);
        map.set('Hangul_EU', 0xed1);
        map.set('Hangul_YI', 0xed2);
        map.set('Hangul_I', 0xed3);
        map.set('Hangul_J_Kiyeog', 0xed4);
        map.set('Hangul_J_SsangKiyeog', 0xed5);
        map.set('Hangul_J_KiyeogSios', 0xed6);
        map.set('Hangul_J_Nieun', 0xed7);
        map.set('Hangul_J_NieunJieuj', 0xed8);
        map.set('Hangul_J_NieunHieuh', 0xed9);
        map.set('Hangul_J_Dikeud', 0xeda);
        map.set('Hangul_J_Rieul', 0xedb);
        map.set('Hangul_J_RieulKiyeog', 0xedc);
        map.set('Hangul_J_RieulMieum', 0xedd);
        map.set('Hangul_J_RieulPieub', 0xede);
        map.set('Hangul_J_RieulSios', 0xedf);
        map.set('Hangul_J_RieulTieut', 0xee0);
        map.set('Hangul_J_RieulPhieuf', 0xee1);
        map.set('Hangul_J_RieulHieuh', 0xee2);
        map.set('Hangul_J_Mieum', 0xee3);
        map.set('Hangul_J_Pieub', 0xee4);
        map.set('Hangul_J_PieubSios', 0xee5);
        map.set('Hangul_J_Sios', 0xee6);
        map.set('Hangul_J_SsangSios', 0xee7);
        map.set('Hangul_J_Ieung', 0xee8);
        map.set('Hangul_J_Jieuj', 0xee9);
        map.set('Hangul_J_Cieuc', 0xeea);
        map.set('Hangul_J_Khieuq', 0xeeb);
        map.set('Hangul_J_Tieut', 0xeec);
        map.set('Hangul_J_Phieuf', 0xeed);
        map.set('Hangul_J_Hieuh', 0xeee);
        map.set('Hangul_RieulYeorinHieuh', 0xeef);
        map.set('Hangul_SunkyeongeumMieum', 0xef0);
        map.set('Hangul_SunkyeongeumPieub', 0xef1);
        map.set('Hangul_PanSios', 0xef2);
        map.set('Hangul_KkogjiDalrinIeung', 0xef3);
        map.set('Hangul_SunkyeongeumPhieuf', 0xef4);
        map.set('Hangul_YeorinHieuh', 0xef5);
        map.set('Hangul_AraeA', 0xef6);
        map.set('Hangul_AraeAE', 0xef7);
        map.set('Hangul_J_PanSios', 0xef8);
        map.set('Hangul_J_KkogjiDalrinIeung', 0xef9);
        map.set('Hangul_J_YeorinHieuh', 0xefa);
        map.set('Korean_Won', 0xeff);
        map.set('Armenian_ligature_ew', 0x1000587);
        map.set('Armenian_full_stop', 0x1000589);
        map.set('Armenian_verjaket', 0x1000589);
        map.set('Armenian_separation_mark', 0x100055d);
        map.set('Armenian_but', 0x100055d);
        map.set('Armenian_hyphen', 0x100058a);
        map.set('Armenian_yentamna', 0x100058a);
        map.set('Armenian_exclam', 0x100055c);
        map.set('Armenian_amanak', 0x100055c);
        map.set('Armenian_accent', 0x100055b);
        map.set('Armenian_shesht', 0x100055b);
        map.set('Armenian_question', 0x100055e);
        map.set('Armenian_paruyk', 0x100055e);
        map.set('Armenian_AYB', 0x1000531);
        map.set('Armenian_ayb', 0x1000561);
        map.set('Armenian_BEN', 0x1000532);
        map.set('Armenian_ben', 0x1000562);
        map.set('Armenian_GIM', 0x1000533);
        map.set('Armenian_gim', 0x1000563);
        map.set('Armenian_DA', 0x1000534);
        map.set('Armenian_da', 0x1000564);
        map.set('Armenian_YECH', 0x1000535);
        map.set('Armenian_yech', 0x1000565);
        map.set('Armenian_ZA', 0x1000536);
        map.set('Armenian_za', 0x1000566);
        map.set('Armenian_E', 0x1000537);
        map.set('Armenian_e', 0x1000567);
        map.set('Armenian_AT', 0x1000538);
        map.set('Armenian_at', 0x1000568);
        map.set('Armenian_TO', 0x1000539);
        map.set('Armenian_to', 0x1000569);
        map.set('Armenian_ZHE', 0x100053a);
        map.set('Armenian_zhe', 0x100056a);
        map.set('Armenian_INI', 0x100053b);
        map.set('Armenian_ini', 0x100056b);
        map.set('Armenian_LYUN', 0x100053c);
        map.set('Armenian_lyun', 0x100056c);
        map.set('Armenian_KHE', 0x100053d);
        map.set('Armenian_khe', 0x100056d);
        map.set('Armenian_TSA', 0x100053e);
        map.set('Armenian_tsa', 0x100056e);
        map.set('Armenian_KEN', 0x100053f);
        map.set('Armenian_ken', 0x100056f);
        map.set('Armenian_HO', 0x1000540);
        map.set('Armenian_ho', 0x1000570);
        map.set('Armenian_DZA', 0x1000541);
        map.set('Armenian_dza', 0x1000571);
        map.set('Armenian_GHAT', 0x1000542);
        map.set('Armenian_ghat', 0x1000572);
        map.set('Armenian_TCHE', 0x1000543);
        map.set('Armenian_tche', 0x1000573);
        map.set('Armenian_MEN', 0x1000544);
        map.set('Armenian_men', 0x1000574);
        map.set('Armenian_HI', 0x1000545);
        map.set('Armenian_hi', 0x1000575);
        map.set('Armenian_NU', 0x1000546);
        map.set('Armenian_nu', 0x1000576);
        map.set('Armenian_SHA', 0x1000547);
        map.set('Armenian_sha', 0x1000577);
        map.set('Armenian_VO', 0x1000548);
        map.set('Armenian_vo', 0x1000578);
        map.set('Armenian_CHA', 0x1000549);
        map.set('Armenian_cha', 0x1000579);
        map.set('Armenian_PE', 0x100054a);
        map.set('Armenian_pe', 0x100057a);
        map.set('Armenian_JE', 0x100054b);
        map.set('Armenian_je', 0x100057b);
        map.set('Armenian_RA', 0x100054c);
        map.set('Armenian_ra', 0x100057c);
        map.set('Armenian_SE', 0x100054d);
        map.set('Armenian_se', 0x100057d);
        map.set('Armenian_VEV', 0x100054e);
        map.set('Armenian_vev', 0x100057e);
        map.set('Armenian_TYUN', 0x100054f);
        map.set('Armenian_tyun', 0x100057f);
        map.set('Armenian_RE', 0x1000550);
        map.set('Armenian_re', 0x1000580);
        map.set('Armenian_TSO', 0x1000551);
        map.set('Armenian_tso', 0x1000581);
        map.set('Armenian_VYUN', 0x1000552);
        map.set('Armenian_vyun', 0x1000582);
        map.set('Armenian_PYUR', 0x1000553);
        map.set('Armenian_pyur', 0x1000583);
        map.set('Armenian_KE', 0x1000554);
        map.set('Armenian_ke', 0x1000584);
        map.set('Armenian_O', 0x1000555);
        map.set('Armenian_o', 0x1000585);
        map.set('Armenian_FE', 0x1000556);
        map.set('Armenian_fe', 0x1000586);
        map.set('Armenian_apostrophe', 0x100055a);
        map.set('Georgian_an', 0x10010d0);
        map.set('Georgian_ban', 0x10010d1);
        map.set('Georgian_gan', 0x10010d2);
        map.set('Georgian_don', 0x10010d3);
        map.set('Georgian_en', 0x10010d4);
        map.set('Georgian_vin', 0x10010d5);
        map.set('Georgian_zen', 0x10010d6);
        map.set('Georgian_tan', 0x10010d7);
        map.set('Georgian_in', 0x10010d8);
        map.set('Georgian_kan', 0x10010d9);
        map.set('Georgian_las', 0x10010da);
        map.set('Georgian_man', 0x10010db);
        map.set('Georgian_nar', 0x10010dc);
        map.set('Georgian_on', 0x10010dd);
        map.set('Georgian_par', 0x10010de);
        map.set('Georgian_zhar', 0x10010df);
        map.set('Georgian_rae', 0x10010e0);
        map.set('Georgian_san', 0x10010e1);
        map.set('Georgian_tar', 0x10010e2);
        map.set('Georgian_un', 0x10010e3);
        map.set('Georgian_phar', 0x10010e4);
        map.set('Georgian_khar', 0x10010e5);
        map.set('Georgian_ghan', 0x10010e6);
        map.set('Georgian_qar', 0x10010e7);
        map.set('Georgian_shin', 0x10010e8);
        map.set('Georgian_chin', 0x10010e9);
        map.set('Georgian_can', 0x10010ea);
        map.set('Georgian_jil', 0x10010eb);
        map.set('Georgian_cil', 0x10010ec);
        map.set('Georgian_char', 0x10010ed);
        map.set('Georgian_xan', 0x10010ee);
        map.set('Georgian_jhan', 0x10010ef);
        map.set('Georgian_hae', 0x10010f0);
        map.set('Georgian_he', 0x10010f1);
        map.set('Georgian_hie', 0x10010f2);
        map.set('Georgian_we', 0x10010f3);
        map.set('Georgian_har', 0x10010f4);
        map.set('Georgian_hoe', 0x10010f5);
        map.set('Georgian_fi', 0x10010f6);
        map.set('Xabovedot', 0x1001e8a);
        map.set('Ibreve', 0x100012c);
        map.set('Zstroke', 0x10001b5);
        map.set('Gcaron', 0x10001e6);
        map.set('Ocaron', 0x10001d1);
        map.set('Obarred', 0x100019f);
        map.set('xabovedot', 0x1001e8b);
        map.set('ibreve', 0x100012d);
        map.set('zstroke', 0x10001b6);
        map.set('gcaron', 0x10001e7);
        map.set('ocaron', 0x10001d2);
        map.set('obarred', 0x1000275);
        map.set('SCHWA', 0x100018f);
        map.set('schwa', 0x1000259);
        map.set('EZH', 0x10001b7);
        map.set('ezh', 0x1000292);
        map.set('Lbelowdot', 0x1001e36);
        map.set('lbelowdot', 0x1001e37);
        map.set('Abelowdot', 0x1001ea0);
        map.set('abelowdot', 0x1001ea1);
        map.set('Ahook', 0x1001ea2);
        map.set('ahook', 0x1001ea3);
        map.set('Acircumflexacute', 0x1001ea4);
        map.set('acircumflexacute', 0x1001ea5);
        map.set('Acircumflexgrave', 0x1001ea6);
        map.set('acircumflexgrave', 0x1001ea7);
        map.set('Acircumflexhook', 0x1001ea8);
        map.set('acircumflexhook', 0x1001ea9);
        map.set('Acircumflextilde', 0x1001eaa);
        map.set('acircumflextilde', 0x1001eab);
        map.set('Acircumflexbelowdot', 0x1001eac);
        map.set('acircumflexbelowdot', 0x1001ead);
        map.set('Abreveacute', 0x1001eae);
        map.set('abreveacute', 0x1001eaf);
        map.set('Abrevegrave', 0x1001eb0);
        map.set('abrevegrave', 0x1001eb1);
        map.set('Abrevehook', 0x1001eb2);
        map.set('abrevehook', 0x1001eb3);
        map.set('Abrevetilde', 0x1001eb4);
        map.set('abrevetilde', 0x1001eb5);
        map.set('Abrevebelowdot', 0x1001eb6);
        map.set('abrevebelowdot', 0x1001eb7);
        map.set('Ebelowdot', 0x1001eb8);
        map.set('ebelowdot', 0x1001eb9);
        map.set('Ehook', 0x1001eba);
        map.set('ehook', 0x1001ebb);
        map.set('Etilde', 0x1001ebc);
        map.set('etilde', 0x1001ebd);
        map.set('Ecircumflexacute', 0x1001ebe);
        map.set('ecircumflexacute', 0x1001ebf);
        map.set('Ecircumflexgrave', 0x1001ec0);
        map.set('ecircumflexgrave', 0x1001ec1);
        map.set('Ecircumflexhook', 0x1001ec2);
        map.set('ecircumflexhook', 0x1001ec3);
        map.set('Ecircumflextilde', 0x1001ec4);
        map.set('ecircumflextilde', 0x1001ec5);
        map.set('Ecircumflexbelowdot', 0x1001ec6);
        map.set('ecircumflexbelowdot', 0x1001ec7);
        map.set('Ihook', 0x1001ec8);
        map.set('ihook', 0x1001ec9);
        map.set('Ibelowdot', 0x1001eca);
        map.set('ibelowdot', 0x1001ecb);
        map.set('Obelowdot', 0x1001ecc);
        map.set('obelowdot', 0x1001ecd);
        map.set('Ohook', 0x1001ece);
        map.set('ohook', 0x1001ecf);
        map.set('Ocircumflexacute', 0x1001ed0);
        map.set('ocircumflexacute', 0x1001ed1);
        map.set('Ocircumflexgrave', 0x1001ed2);
        map.set('ocircumflexgrave', 0x1001ed3);
        map.set('Ocircumflexhook', 0x1001ed4);
        map.set('ocircumflexhook', 0x1001ed5);
        map.set('Ocircumflextilde', 0x1001ed6);
        map.set('ocircumflextilde', 0x1001ed7);
        map.set('Ocircumflexbelowdot', 0x1001ed8);
        map.set('ocircumflexbelowdot', 0x1001ed9);
        map.set('Ohornacute', 0x1001eda);
        map.set('ohornacute', 0x1001edb);
        map.set('Ohorngrave', 0x1001edc);
        map.set('ohorngrave', 0x1001edd);
        map.set('Ohornhook', 0x1001ede);
        map.set('ohornhook', 0x1001edf);
        map.set('Ohorntilde', 0x1001ee0);
        map.set('ohorntilde', 0x1001ee1);
        map.set('Ohornbelowdot', 0x1001ee2);
        map.set('ohornbelowdot', 0x1001ee3);
        map.set('Ubelowdot', 0x1001ee4);
        map.set('ubelowdot', 0x1001ee5);
        map.set('Uhook', 0x1001ee6);
        map.set('uhook', 0x1001ee7);
        map.set('Uhornacute', 0x1001ee8);
        map.set('uhornacute', 0x1001ee9);
        map.set('Uhorngrave', 0x1001eea);
        map.set('uhorngrave', 0x1001eeb);
        map.set('Uhornhook', 0x1001eec);
        map.set('uhornhook', 0x1001eed);
        map.set('Uhorntilde', 0x1001eee);
        map.set('uhorntilde', 0x1001eef);
        map.set('Uhornbelowdot', 0x1001ef0);
        map.set('uhornbelowdot', 0x1001ef1);
        map.set('Ybelowdot', 0x1001ef4);
        map.set('ybelowdot', 0x1001ef5);
        map.set('Yhook', 0x1001ef6);
        map.set('yhook', 0x1001ef7);
        map.set('Ytilde', 0x1001ef8);
        map.set('ytilde', 0x1001ef9);
        map.set('Ohorn', 0x10001a0);
        map.set('ohorn', 0x10001a1);
        map.set('Uhorn', 0x10001af);
        map.set('uhorn', 0x10001b0);
        map.set('EcuSign', 0x10020a0);
        map.set('ColonSign', 0x10020a1);
        map.set('CruzeiroSign', 0x10020a2);
        map.set('FFrancSign', 0x10020a3);
        map.set('LiraSign', 0x10020a4);
        map.set('MillSign', 0x10020a5);
        map.set('NairaSign', 0x10020a6);
        map.set('PesetaSign', 0x10020a7);
        map.set('RupeeSign', 0x10020a8);
        map.set('WonSign', 0x10020a9);
        map.set('NewSheqelSign', 0x10020aa);
        map.set('DongSign', 0x10020ab);
        map.set('EuroSign', 0x20ac);
        map.set('zerosuperior', 0x1002070);
        map.set('foursuperior', 0x1002074);
        map.set('fivesuperior', 0x1002075);
        map.set('sixsuperior', 0x1002076);
        map.set('sevensuperior', 0x1002077);
        map.set('eightsuperior', 0x1002078);
        map.set('ninesuperior', 0x1002079);
        map.set('zerosubscript', 0x1002080);
        map.set('onesubscript', 0x1002081);
        map.set('twosubscript', 0x1002082);
        map.set('threesubscript', 0x1002083);
        map.set('foursubscript', 0x1002084);
        map.set('fivesubscript', 0x1002085);
        map.set('sixsubscript', 0x1002086);
        map.set('sevensubscript', 0x1002087);
        map.set('eightsubscript', 0x1002088);
        map.set('ninesubscript', 0x1002089);
        map.set('partdifferential', 0x1002202);
        map.set('emptyset', 0x1002205);
        map.set('elementof', 0x1002208);
        map.set('notelementof', 0x1002209);
        map.set('containsas', 0x100220b);
        map.set('squareroot', 0x100221a);
        map.set('cuberoot', 0x100221b);
        map.set('fourthroot', 0x100221c);
        map.set('dintegral', 0x100222c);
        map.set('tintegral', 0x100222d);
        map.set('because', 0x1002235);
        map.set('approxeq', 0x1002248);
        map.set('notapproxeq', 0x1002247);
        map.set('notidentical', 0x1002262);
        map.set('stricteq', 0x1002263);
        map.set('braille_dot_1', 0xfff1);
        map.set('braille_dot_2', 0xfff2);
        map.set('braille_dot_3', 0xfff3);
        map.set('braille_dot_4', 0xfff4);
        map.set('braille_dot_5', 0xfff5);
        map.set('braille_dot_6', 0xfff6);
        map.set('braille_dot_7', 0xfff7);
        map.set('braille_dot_8', 0xfff8);
        map.set('braille_dot_9', 0xfff9);
        map.set('braille_dot_10', 0xfffa);
        map.set('braille_blank', 0x1002800);
        map.set('braille_dots_1', 0x1002801);
        map.set('braille_dots_2', 0x1002802);
        map.set('braille_dots_12', 0x1002803);
        map.set('braille_dots_3', 0x1002804);
        map.set('braille_dots_13', 0x1002805);
        map.set('braille_dots_23', 0x1002806);
        map.set('braille_dots_123', 0x1002807);
        map.set('braille_dots_4', 0x1002808);
        map.set('braille_dots_14', 0x1002809);
        map.set('braille_dots_24', 0x100280a);
        map.set('braille_dots_124', 0x100280b);
        map.set('braille_dots_34', 0x100280c);
        map.set('braille_dots_134', 0x100280d);
        map.set('braille_dots_234', 0x100280e);
        map.set('braille_dots_1234', 0x100280f);
        map.set('braille_dots_5', 0x1002810);
        map.set('braille_dots_15', 0x1002811);
        map.set('braille_dots_25', 0x1002812);
        map.set('braille_dots_125', 0x1002813);
        map.set('braille_dots_35', 0x1002814);
        map.set('braille_dots_135', 0x1002815);
        map.set('braille_dots_235', 0x1002816);
        map.set('braille_dots_1235', 0x1002817);
        map.set('braille_dots_45', 0x1002818);
        map.set('braille_dots_145', 0x1002819);
        map.set('braille_dots_245', 0x100281a);
        map.set('braille_dots_1245', 0x100281b);
        map.set('braille_dots_345', 0x100281c);
        map.set('braille_dots_1345', 0x100281d);
        map.set('braille_dots_2345', 0x100281e);
        map.set('braille_dots_12345', 0x100281f);
        map.set('braille_dots_6', 0x1002820);
        map.set('braille_dots_16', 0x1002821);
        map.set('braille_dots_26', 0x1002822);
        map.set('braille_dots_126', 0x1002823);
        map.set('braille_dots_36', 0x1002824);
        map.set('braille_dots_136', 0x1002825);
        map.set('braille_dots_236', 0x1002826);
        map.set('braille_dots_1236', 0x1002827);
        map.set('braille_dots_46', 0x1002828);
        map.set('braille_dots_146', 0x1002829);
        map.set('braille_dots_246', 0x100282a);
        map.set('braille_dots_1246', 0x100282b);
        map.set('braille_dots_346', 0x100282c);
        map.set('braille_dots_1346', 0x100282d);
        map.set('braille_dots_2346', 0x100282e);
        map.set('braille_dots_12346', 0x100282f);
        map.set('braille_dots_56', 0x1002830);
        map.set('braille_dots_156', 0x1002831);
        map.set('braille_dots_256', 0x1002832);
        map.set('braille_dots_1256', 0x1002833);
        map.set('braille_dots_356', 0x1002834);
        map.set('braille_dots_1356', 0x1002835);
        map.set('braille_dots_2356', 0x1002836);
        map.set('braille_dots_12356', 0x1002837);
        map.set('braille_dots_456', 0x1002838);
        map.set('braille_dots_1456', 0x1002839);
        map.set('braille_dots_2456', 0x100283a);
        map.set('braille_dots_12456', 0x100283b);
        map.set('braille_dots_3456', 0x100283c);
        map.set('braille_dots_13456', 0x100283d);
        map.set('braille_dots_23456', 0x100283e);
        map.set('braille_dots_123456', 0x100283f);
        map.set('braille_dots_7', 0x1002840);
        map.set('braille_dots_17', 0x1002841);
        map.set('braille_dots_27', 0x1002842);
        map.set('braille_dots_127', 0x1002843);
        map.set('braille_dots_37', 0x1002844);
        map.set('braille_dots_137', 0x1002845);
        map.set('braille_dots_237', 0x1002846);
        map.set('braille_dots_1237', 0x1002847);
        map.set('braille_dots_47', 0x1002848);
        map.set('braille_dots_147', 0x1002849);
        map.set('braille_dots_247', 0x100284a);
        map.set('braille_dots_1247', 0x100284b);
        map.set('braille_dots_347', 0x100284c);
        map.set('braille_dots_1347', 0x100284d);
        map.set('braille_dots_2347', 0x100284e);
        map.set('braille_dots_12347', 0x100284f);
        map.set('braille_dots_57', 0x1002850);
        map.set('braille_dots_157', 0x1002851);
        map.set('braille_dots_257', 0x1002852);
        map.set('braille_dots_1257', 0x1002853);
        map.set('braille_dots_357', 0x1002854);
        map.set('braille_dots_1357', 0x1002855);
        map.set('braille_dots_2357', 0x1002856);
        map.set('braille_dots_12357', 0x1002857);
        map.set('braille_dots_457', 0x1002858);
        map.set('braille_dots_1457', 0x1002859);
        map.set('braille_dots_2457', 0x100285a);
        map.set('braille_dots_12457', 0x100285b);
        map.set('braille_dots_3457', 0x100285c);
        map.set('braille_dots_13457', 0x100285d);
        map.set('braille_dots_23457', 0x100285e);
        map.set('braille_dots_123457', 0x100285f);
        map.set('braille_dots_67', 0x1002860);
        map.set('braille_dots_167', 0x1002861);
        map.set('braille_dots_267', 0x1002862);
        map.set('braille_dots_1267', 0x1002863);
        map.set('braille_dots_367', 0x1002864);
        map.set('braille_dots_1367', 0x1002865);
        map.set('braille_dots_2367', 0x1002866);
        map.set('braille_dots_12367', 0x1002867);
        map.set('braille_dots_467', 0x1002868);
        map.set('braille_dots_1467', 0x1002869);
        map.set('braille_dots_2467', 0x100286a);
        map.set('braille_dots_12467', 0x100286b);
        map.set('braille_dots_3467', 0x100286c);
        map.set('braille_dots_13467', 0x100286d);
        map.set('braille_dots_23467', 0x100286e);
        map.set('braille_dots_123467', 0x100286f);
        map.set('braille_dots_567', 0x1002870);
        map.set('braille_dots_1567', 0x1002871);
        map.set('braille_dots_2567', 0x1002872);
        map.set('braille_dots_12567', 0x1002873);
        map.set('braille_dots_3567', 0x1002874);
        map.set('braille_dots_13567', 0x1002875);
        map.set('braille_dots_23567', 0x1002876);
        map.set('braille_dots_123567', 0x1002877);
        map.set('braille_dots_4567', 0x1002878);
        map.set('braille_dots_14567', 0x1002879);
        map.set('braille_dots_24567', 0x100287a);
        map.set('braille_dots_124567', 0x100287b);
        map.set('braille_dots_34567', 0x100287c);
        map.set('braille_dots_134567', 0x100287d);
        map.set('braille_dots_234567', 0x100287e);
        map.set('braille_dots_1234567', 0x100287f);
        map.set('braille_dots_8', 0x1002880);
        map.set('braille_dots_18', 0x1002881);
        map.set('braille_dots_28', 0x1002882);
        map.set('braille_dots_128', 0x1002883);
        map.set('braille_dots_38', 0x1002884);
        map.set('braille_dots_138', 0x1002885);
        map.set('braille_dots_238', 0x1002886);
        map.set('braille_dots_1238', 0x1002887);
        map.set('braille_dots_48', 0x1002888);
        map.set('braille_dots_148', 0x1002889);
        map.set('braille_dots_248', 0x100288a);
        map.set('braille_dots_1248', 0x100288b);
        map.set('braille_dots_348', 0x100288c);
        map.set('braille_dots_1348', 0x100288d);
        map.set('braille_dots_2348', 0x100288e);
        map.set('braille_dots_12348', 0x100288f);
        map.set('braille_dots_58', 0x1002890);
        map.set('braille_dots_158', 0x1002891);
        map.set('braille_dots_258', 0x1002892);
        map.set('braille_dots_1258', 0x1002893);
        map.set('braille_dots_358', 0x1002894);
        map.set('braille_dots_1358', 0x1002895);
        map.set('braille_dots_2358', 0x1002896);
        map.set('braille_dots_12358', 0x1002897);
        map.set('braille_dots_458', 0x1002898);
        map.set('braille_dots_1458', 0x1002899);
        map.set('braille_dots_2458', 0x100289a);
        map.set('braille_dots_12458', 0x100289b);
        map.set('braille_dots_3458', 0x100289c);
        map.set('braille_dots_13458', 0x100289d);
        map.set('braille_dots_23458', 0x100289e);
        map.set('braille_dots_123458', 0x100289f);
        map.set('braille_dots_68', 0x10028a0);
        map.set('braille_dots_168', 0x10028a1);
        map.set('braille_dots_268', 0x10028a2);
        map.set('braille_dots_1268', 0x10028a3);
        map.set('braille_dots_368', 0x10028a4);
        map.set('braille_dots_1368', 0x10028a5);
        map.set('braille_dots_2368', 0x10028a6);
        map.set('braille_dots_12368', 0x10028a7);
        map.set('braille_dots_468', 0x10028a8);
        map.set('braille_dots_1468', 0x10028a9);
        map.set('braille_dots_2468', 0x10028aa);
        map.set('braille_dots_12468', 0x10028ab);
        map.set('braille_dots_3468', 0x10028ac);
        map.set('braille_dots_13468', 0x10028ad);
        map.set('braille_dots_23468', 0x10028ae);
        map.set('braille_dots_123468', 0x10028af);
        map.set('braille_dots_568', 0x10028b0);
        map.set('braille_dots_1568', 0x10028b1);
        map.set('braille_dots_2568', 0x10028b2);
        map.set('braille_dots_12568', 0x10028b3);
        map.set('braille_dots_3568', 0x10028b4);
        map.set('braille_dots_13568', 0x10028b5);
        map.set('braille_dots_23568', 0x10028b6);
        map.set('braille_dots_123568', 0x10028b7);
        map.set('braille_dots_4568', 0x10028b8);
        map.set('braille_dots_14568', 0x10028b9);
        map.set('braille_dots_24568', 0x10028ba);
        map.set('braille_dots_124568', 0x10028bb);
        map.set('braille_dots_34568', 0x10028bc);
        map.set('braille_dots_134568', 0x10028bd);
        map.set('braille_dots_234568', 0x10028be);
        map.set('braille_dots_1234568', 0x10028bf);
        map.set('braille_dots_78', 0x10028c0);
        map.set('braille_dots_178', 0x10028c1);
        map.set('braille_dots_278', 0x10028c2);
        map.set('braille_dots_1278', 0x10028c3);
        map.set('braille_dots_378', 0x10028c4);
        map.set('braille_dots_1378', 0x10028c5);
        map.set('braille_dots_2378', 0x10028c6);
        map.set('braille_dots_12378', 0x10028c7);
        map.set('braille_dots_478', 0x10028c8);
        map.set('braille_dots_1478', 0x10028c9);
        map.set('braille_dots_2478', 0x10028ca);
        map.set('braille_dots_12478', 0x10028cb);
        map.set('braille_dots_3478', 0x10028cc);
        map.set('braille_dots_13478', 0x10028cd);
        map.set('braille_dots_23478', 0x10028ce);
        map.set('braille_dots_123478', 0x10028cf);
        map.set('braille_dots_578', 0x10028d0);
        map.set('braille_dots_1578', 0x10028d1);
        map.set('braille_dots_2578', 0x10028d2);
        map.set('braille_dots_12578', 0x10028d3);
        map.set('braille_dots_3578', 0x10028d4);
        map.set('braille_dots_13578', 0x10028d5);
        map.set('braille_dots_23578', 0x10028d6);
        map.set('braille_dots_123578', 0x10028d7);
        map.set('braille_dots_4578', 0x10028d8);
        map.set('braille_dots_14578', 0x10028d9);
        map.set('braille_dots_24578', 0x10028da);
        map.set('braille_dots_124578', 0x10028db);
        map.set('braille_dots_34578', 0x10028dc);
        map.set('braille_dots_134578', 0x10028dd);
        map.set('braille_dots_234578', 0x10028de);
        map.set('braille_dots_1234578', 0x10028df);
        map.set('braille_dots_678', 0x10028e0);
        map.set('braille_dots_1678', 0x10028e1);
        map.set('braille_dots_2678', 0x10028e2);
        map.set('braille_dots_12678', 0x10028e3);
        map.set('braille_dots_3678', 0x10028e4);
        map.set('braille_dots_13678', 0x10028e5);
        map.set('braille_dots_23678', 0x10028e6);
        map.set('braille_dots_123678', 0x10028e7);
        map.set('braille_dots_4678', 0x10028e8);
        map.set('braille_dots_14678', 0x10028e9);
        map.set('braille_dots_24678', 0x10028ea);
        map.set('braille_dots_124678', 0x10028eb);
        map.set('braille_dots_34678', 0x10028ec);
        map.set('braille_dots_134678', 0x10028ed);
        map.set('braille_dots_234678', 0x10028ee);
        map.set('braille_dots_1234678', 0x10028ef);
        map.set('braille_dots_5678', 0x10028f0);
        map.set('braille_dots_15678', 0x10028f1);
        map.set('braille_dots_25678', 0x10028f2);
        map.set('braille_dots_125678', 0x10028f3);
        map.set('braille_dots_35678', 0x10028f4);
        map.set('braille_dots_135678', 0x10028f5);
        map.set('braille_dots_235678', 0x10028f6);
        map.set('braille_dots_1235678', 0x10028f7);
        map.set('braille_dots_45678', 0x10028f8);
        map.set('braille_dots_145678', 0x10028f9);
        map.set('braille_dots_245678', 0x10028fa);
        map.set('braille_dots_1245678', 0x10028fb);
        map.set('braille_dots_345678', 0x10028fc);
        map.set('braille_dots_1345678', 0x10028fd);
        map.set('braille_dots_2345678', 0x10028fe);
        map.set('braille_dots_12345678', 0x10028ff);
        map.set('Sinh_ng', 0x1000d82);
        map.set('Sinh_h2', 0x1000d83);
        map.set('Sinh_a', 0x1000d85);
        map.set('Sinh_aa', 0x1000d86);
        map.set('Sinh_ae', 0x1000d87);
        map.set('Sinh_aee', 0x1000d88);
        map.set('Sinh_i', 0x1000d89);
        map.set('Sinh_ii', 0x1000d8a);
        map.set('Sinh_u', 0x1000d8b);
        map.set('Sinh_uu', 0x1000d8c);
        map.set('Sinh_ri', 0x1000d8d);
        map.set('Sinh_rii', 0x1000d8e);
        map.set('Sinh_lu', 0x1000d8f);
        map.set('Sinh_luu', 0x1000d90);
        map.set('Sinh_e', 0x1000d91);
        map.set('Sinh_ee', 0x1000d92);
        map.set('Sinh_ai', 0x1000d93);
        map.set('Sinh_o', 0x1000d94);
        map.set('Sinh_oo', 0x1000d95);
        map.set('Sinh_au', 0x1000d96);
        map.set('Sinh_ka', 0x1000d9a);
        map.set('Sinh_kha', 0x1000d9b);
        map.set('Sinh_ga', 0x1000d9c);
        map.set('Sinh_gha', 0x1000d9d);
        map.set('Sinh_ng2', 0x1000d9e);
        map.set('Sinh_nga', 0x1000d9f);
        map.set('Sinh_ca', 0x1000da0);
        map.set('Sinh_cha', 0x1000da1);
        map.set('Sinh_ja', 0x1000da2);
        map.set('Sinh_jha', 0x1000da3);
        map.set('Sinh_nya', 0x1000da4);
        map.set('Sinh_jnya', 0x1000da5);
        map.set('Sinh_nja', 0x1000da6);
        map.set('Sinh_tta', 0x1000da7);
        map.set('Sinh_ttha', 0x1000da8);
        map.set('Sinh_dda', 0x1000da9);
        map.set('Sinh_ddha', 0x1000daa);
        map.set('Sinh_nna', 0x1000dab);
        map.set('Sinh_ndda', 0x1000dac);
        map.set('Sinh_tha', 0x1000dad);
        map.set('Sinh_thha', 0x1000dae);
        map.set('Sinh_dha', 0x1000daf);
        map.set('Sinh_dhha', 0x1000db0);
        map.set('Sinh_na', 0x1000db1);
        map.set('Sinh_ndha', 0x1000db3);
        map.set('Sinh_pa', 0x1000db4);
        map.set('Sinh_pha', 0x1000db5);
        map.set('Sinh_ba', 0x1000db6);
        map.set('Sinh_bha', 0x1000db7);
        map.set('Sinh_ma', 0x1000db8);
        map.set('Sinh_mba', 0x1000db9);
        map.set('Sinh_ya', 0x1000dba);
        map.set('Sinh_ra', 0x1000dbb);
        map.set('Sinh_la', 0x1000dbd);
        map.set('Sinh_va', 0x1000dc0);
        map.set('Sinh_sha', 0x1000dc1);
        map.set('Sinh_ssha', 0x1000dc2);
        map.set('Sinh_sa', 0x1000dc3);
        map.set('Sinh_ha', 0x1000dc4);
        map.set('Sinh_lla', 0x1000dc5);
        map.set('Sinh_fa', 0x1000dc6);
        map.set('Sinh_al', 0x1000dca);
        map.set('Sinh_aa2', 0x1000dcf);
        map.set('Sinh_ae2', 0x1000dd0);
        map.set('Sinh_aee2', 0x1000dd1);
        map.set('Sinh_i2', 0x1000dd2);
        map.set('Sinh_ii2', 0x1000dd3);
        map.set('Sinh_u2', 0x1000dd4);
        map.set('Sinh_uu2', 0x1000dd6);
        map.set('Sinh_ru2', 0x1000dd8);
        map.set('Sinh_e2', 0x1000dd9);
        map.set('Sinh_ee2', 0x1000dda);
        map.set('Sinh_ai2', 0x1000ddb);
        map.set('Sinh_o2', 0x1000ddc);
        map.set('Sinh_oo2', 0x1000ddd);
        map.set('Sinh_au2', 0x1000dde);
        map.set('Sinh_lu2', 0x1000ddf);
        map.set('Sinh_ruu2', 0x1000df2);
        map.set('Sinh_luu2', 0x1000df3);
        map.set('Sinh_kunddaliya', 0x1000df4);
        map.set('ModeLock', 0x1008ff01);
        map.set('MonBrightnessUp', 0x1008ff02);
        map.set('MonBrightnessDown', 0x1008ff03);
        map.set('KbdLightOnOff', 0x1008ff04);
        map.set('KbdBrightnessUp', 0x1008ff05);
        map.set('KbdBrightnessDown', 0x1008ff06);
        map.set('Standby', 0x1008ff10);
        map.set('AudioLowerVolume', 0x1008ff11);
        map.set('AudioMute', 0x1008ff12);
        map.set('AudioRaiseVolume', 0x1008ff13);
        map.set('AudioPlay', 0x1008ff14);
        map.set('AudioStop', 0x1008ff15);
        map.set('AudioPrev', 0x1008ff16);
        map.set('AudioNext', 0x1008ff17);
        map.set('HomePage', 0x1008ff18);
        map.set('Mail', 0x1008ff19);
        map.set('Start', 0x1008ff1a);
        map.set('Search', 0x1008ff1b);
        map.set('AudioRecord', 0x1008ff1c);
        map.set('Calculator', 0x1008ff1d);
        map.set('Memo', 0x1008ff1e);
        map.set('ToDoList', 0x1008ff1f);
        map.set('Calendar', 0x1008ff20);
        map.set('PowerDown', 0x1008ff21);
        map.set('ContrastAdjust', 0x1008ff22);
        map.set('RockerUp', 0x1008ff23);
        map.set('RockerDown', 0x1008ff24);
        map.set('RockerEnter', 0x1008ff25);
        map.set('Back', 0x1008ff26);
        map.set('Forward', 0x1008ff27);
        map.set('Stop', 0x1008ff28);
        map.set('Refresh', 0x1008ff29);
        map.set('PowerOff', 0x1008ff2a);
        map.set('WakeUp', 0x1008ff2b);
        map.set('Eject', 0x1008ff2c);
        map.set('ScreenSaver', 0x1008ff2d);
        map.set('WWW', 0x1008ff2e);
        map.set('Sleep', 0x1008ff2f);
        map.set('Favorites', 0x1008ff30);
        map.set('AudioPause', 0x1008ff31);
        map.set('AudioMedia', 0x1008ff32);
        map.set('MyComputer', 0x1008ff33);
        map.set('VendorHome', 0x1008ff34);
        map.set('LightBulb', 0x1008ff35);
        map.set('Shop', 0x1008ff36);
        map.set('History', 0x1008ff37);
        map.set('OpenURL', 0x1008ff38);
        map.set('AddFavorite', 0x1008ff39);
        map.set('HotLinks', 0x1008ff3a);
        map.set('BrightnessAdjust', 0x1008ff3b);
        map.set('Finance', 0x1008ff3c);
        map.set('Community', 0x1008ff3d);
        map.set('AudioRewind', 0x1008ff3e);
        map.set('BackForward', 0x1008ff3f);
        map.set('Launch0', 0x1008ff40);
        map.set('Launch1', 0x1008ff41);
        map.set('Launch2', 0x1008ff42);
        map.set('Launch3', 0x1008ff43);
        map.set('Launch4', 0x1008ff44);
        map.set('Launch5', 0x1008ff45);
        map.set('Launch6', 0x1008ff46);
        map.set('Launch7', 0x1008ff47);
        map.set('Launch8', 0x1008ff48);
        map.set('Launch9', 0x1008ff49);
        map.set('LaunchA', 0x1008ff4a);
        map.set('LaunchB', 0x1008ff4b);
        map.set('LaunchC', 0x1008ff4c);
        map.set('LaunchD', 0x1008ff4d);
        map.set('LaunchE', 0x1008ff4e);
        map.set('LaunchF', 0x1008ff4f);
        map.set('ApplicationLeft', 0x1008ff50);
        map.set('ApplicationRight', 0x1008ff51);
        map.set('Book', 0x1008ff52);
        map.set('CD', 0x1008ff53);
        map.set('WindowClear', 0x1008ff55);
        map.set('Close', 0x1008ff56);
        map.set('Copy', 0x1008ff57);
        map.set('Cut', 0x1008ff58);
        map.set('Display', 0x1008ff59);
        map.set('DOS', 0x1008ff5a);
        map.set('Documents', 0x1008ff5b);
        map.set('Excel', 0x1008ff5c);
        map.set('Explorer', 0x1008ff5d);
        map.set('Game', 0x1008ff5e);
        map.set('Go', 0x1008ff5f);
        map.set('iTouch', 0x1008ff60);
        map.set('LogOff', 0x1008ff61);
        map.set('Market', 0x1008ff62);
        map.set('Meeting', 0x1008ff63);
        map.set('MenuKB', 0x1008ff65);
        map.set('MenuPB', 0x1008ff66);
        map.set('MySites', 0x1008ff67);
        map.set('New', 0x1008ff68);
        map.set('News', 0x1008ff69);
        map.set('OfficeHome', 0x1008ff6a);
        map.set('Open', 0x1008ff6b);
        map.set('Option', 0x1008ff6c);
        map.set('Paste', 0x1008ff6d);
        map.set('Phone', 0x1008ff6e);
        map.set('Reply', 0x1008ff72);
        map.set('Reload', 0x1008ff73);
        map.set('RotateWindows', 0x1008ff74);
        map.set('RotationPB', 0x1008ff75);
        map.set('RotationKB', 0x1008ff76);
        map.set('Save', 0x1008ff77);
        map.set('ScrollUp', 0x1008ff78);
        map.set('ScrollDown', 0x1008ff79);
        map.set('ScrollClick', 0x1008ff7a);
        map.set('Send', 0x1008ff7b);
        map.set('Spell', 0x1008ff7c);
        map.set('SplitScreen', 0x1008ff7d);
        map.set('Support', 0x1008ff7e);
        map.set('TaskPane', 0x1008ff7f);
        map.set('Terminal', 0x1008ff80);
        map.set('Tools', 0x1008ff81);
        map.set('Travel', 0x1008ff82);
        map.set('UserPB', 0x1008ff84);
        map.set('User1KB', 0x1008ff85);
        map.set('User2KB', 0x1008ff86);
        map.set('Video', 0x1008ff87);
        map.set('WheelButton', 0x1008ff88);
        map.set('Word', 0x1008ff89);
        map.set('Xfer', 0x1008ff8a);
        map.set('ZoomIn', 0x1008ff8b);
        map.set('ZoomOut', 0x1008ff8c);
        map.set('Away', 0x1008ff8d);
        map.set('Messenger', 0x1008ff8e);
        map.set('WebCam', 0x1008ff8f);
        map.set('MailForward', 0x1008ff90);
        map.set('Pictures', 0x1008ff91);
        map.set('Music', 0x1008ff92);
        map.set('Battery', 0x1008ff93);
        map.set('Bluetooth', 0x1008ff94);
        map.set('WLAN', 0x1008ff95);
        map.set('UWB', 0x1008ff96);
        map.set('AudioForward', 0x1008ff97);
        map.set('AudioRepeat', 0x1008ff98);
        map.set('AudioRandomPlay', 0x1008ff99);
        map.set('Subtitle', 0x1008ff9a);
        map.set('AudioCycleTrack', 0x1008ff9b);
        map.set('CycleAngle', 0x1008ff9c);
        map.set('FrameBack', 0x1008ff9d);
        map.set('FrameForward', 0x1008ff9e);
        map.set('Time', 0x1008ff9f);
        map.set('SelectButton', 0x1008ffa0);
        map.set('View', 0x1008ffa1);
        map.set('TopMenu', 0x1008ffa2);
        map.set('Red', 0x1008ffa3);
        map.set('Green', 0x1008ffa4);
        map.set('Yellow', 0x1008ffa5);
        map.set('Blue', 0x1008ffa6);
        map.set('Suspend', 0x1008ffa7);
        map.set('Hibernate', 0x1008ffa8);
        map.set('TouchpadToggle', 0x1008ffa9);
        map.set('TouchpadOn', 0x1008ffb0);
        map.set('TouchpadOff', 0x1008ffb1);
        map.set('AudioMicMute', 0x1008ffb2);
        map.set('Keyboard', 0x1008ffb3);
        map.set('WWAN', 0x1008ffb4);
        map.set('RFKill', 0x1008ffb5);
        map.set('AudioPreset', 0x1008ffb6);
        map.set('Switch_VT_1', 0x1008fe01);
        map.set('Switch_VT_2', 0x1008fe02);
        map.set('Switch_VT_3', 0x1008fe03);
        map.set('Switch_VT_4', 0x1008fe04);
        map.set('Switch_VT_5', 0x1008fe05);
        map.set('Switch_VT_6', 0x1008fe06);
        map.set('Switch_VT_7', 0x1008fe07);
        map.set('Switch_VT_8', 0x1008fe08);
        map.set('Switch_VT_9', 0x1008fe09);
        map.set('Switch_VT_10', 0x1008fe0a);
        map.set('Switch_VT_11', 0x1008fe0b);
        map.set('Switch_VT_12', 0x1008fe0c);
        map.set('Ungrab', 0x1008fe20);
        map.set('ClearGrab', 0x1008fe21);
        map.set('Next_VMode', 0x1008fe22);
        map.set('Prev_VMode', 0x1008fe23);
        map.set('LogWindowTree', 0x1008fe24);
        map.set('LogGrabInfo', 0x1008fe25);
        this.keycodeMap = map;
    }
}
